"""
Uncoder IO Community Edition License
-----------------------------------------------------------------
Copyright (c) 2024 SOC Prime, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
-----------------------------------------------------------------
"""

import copy
import json
from typing import Optional

from app.translator.core.const import QUERY_TOKEN_TYPE
from app.translator.core.custom_types.meta_info import SeverityType
from app.translator.core.mapping import SourceMapping
from app.translator.core.models.platform_details import PlatformDetails
from app.translator.core.models.query_container import MetaInfoContainer, RawQueryContainer, TokenizedQueryContainer
from app.translator.core.models.query_tokens.field_value import FieldValue
from app.translator.managers import render_manager
from app.translator.platforms.base.lucene.mapping import LuceneMappings
from app.translator.platforms.opensearch.const import OPENSEARCH_RULE, opensearch_rule_details
from app.translator.platforms.opensearch.mapping import opensearch_rule_mappings
from app.translator.platforms.opensearch.renders.opensearch import OpenSearchFieldValueRender, OpenSearchQueryRender

_AUTOGENERATED_TEMPLATE = "Autogenerated AWS OpenSearch Rule"
_SEVERITIES_MAP = {SeverityType.critical: "5", SeverityType.high: "4", SeverityType.medium: "3", SeverityType.low: "2"}


class OpenSearchRuleFieldValueRender(OpenSearchFieldValueRender):
    details: PlatformDetails = opensearch_rule_details


@render_manager.register
class OpenSearchRuleRender(OpenSearchQueryRender):
    details: PlatformDetails = opensearch_rule_details
    mappings: LuceneMappings = opensearch_rule_mappings

    or_token = "OR"
    and_token = "AND"
    not_token = "NOT"

    field_value_render = OpenSearchRuleFieldValueRender(or_token=or_token)

    def __init__(self):
        super().__init__()
        self.fields = {}

    def finalize_query(
        self,
        prefix: str,
        query: str,
        functions: str,
        meta_info: Optional[MetaInfoContainer] = None,
        source_mapping: Optional[SourceMapping] = None,  # noqa: ARG002
        not_supported_functions: Optional[list] = None,
        unmapped_fields: Optional[list[str]] = None,
        *args,  # noqa: ARG002
        **kwargs,  # noqa: ARG002
    ) -> str:
        query = super().finalize_query(prefix=prefix, query=query, functions=functions)
        rule = copy.deepcopy(OPENSEARCH_RULE)
        source = {"event.severity": _SEVERITIES_MAP[meta_info.severity], "message": "Alert"}
        source.update(self.fields)
        rule["name"] = meta_info.title or _AUTOGENERATED_TEMPLATE
        rule["inputs"][0]["search"]["query"]["query"]["bool"]["must"][0]["query_string"]["query"] = query
        rule["triggers"][0]["name"] = meta_info.title
        rule["triggers"][0]["severity"] = _SEVERITIES_MAP[meta_info.severity]
        rule["triggers"][0]["actions"][0]["message_template"]["source"] = str(source).replace(", ", ",\n")
        rule_str = json.dumps(rule, indent=4, sort_keys=False)
        rule_str = self.wrap_with_unmapped_fields(rule_str, unmapped_fields)
        return self.wrap_with_not_supported_functions(rule_str, not_supported_functions)

    def apply_token(self, token: QUERY_TOKEN_TYPE, source_mapping: SourceMapping) -> str:
        if isinstance(token, FieldValue) and token.field:
            for field in self.mappings.map_field(token.field, source_mapping):
                self.fields.update({field: f"{{ctx.results.0.hits.hits.0._source.{field}}}"})
        return super().apply_token(token, source_mapping)

    def generate(
        self, raw_query_container: RawQueryContainer, tokenized_query_container: Optional[TokenizedQueryContainer]
    ) -> str:
        self.fields = {}
        return super().generate(
            raw_query_container=raw_query_container, tokenized_query_container=tokenized_query_container
        )
