"""
Uncoder IO Community Edition License
-----------------------------------------------------------------
Copyright (c) 2024 SOC Prime, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
-----------------------------------------------------------------
"""

import copy
import json
from typing import Optional

from app.translator.core.custom_types.meta_info import SeverityType
from app.translator.core.mapping import SourceMapping
from app.translator.core.models.platform_details import PlatformDetails
from app.translator.core.models.query_container import MetaInfoContainer
from app.translator.managers import render_manager
from app.translator.platforms.logrhythm_axon.const import DEFAULT_LOGRHYTHM_AXON_RULE, logrhythm_axon_rule_details
from app.translator.platforms.logrhythm_axon.escape_manager import logrhythm_rule_escape_manager
from app.translator.platforms.logrhythm_axon.mapping import LogRhythmAxonMappings, logrhythm_axon_rule_mappings
from app.translator.platforms.logrhythm_axon.renders.logrhythm_axon_query import (
    LogRhythmAxonFieldValueRender,
    LogRhythmAxonQueryRender,
)
from app.translator.tools.utils import get_rule_description_str

_AUTOGENERATED_TEMPLATE = "Autogenerated LogRhythm Axon Rule"
_SEVERITIES_MAP = {
    SeverityType.critical: SeverityType.critical,
    SeverityType.high: SeverityType.high,
    SeverityType.medium: SeverityType.medium,
    SeverityType.low: SeverityType.low,
    SeverityType.informational: SeverityType.low,
}


class LogRhythmAxonRuleFieldValueRender(LogRhythmAxonFieldValueRender):
    details: PlatformDetails = logrhythm_axon_rule_details
    escape_manager = logrhythm_rule_escape_manager


@render_manager.register
class LogRhythmAxonRuleRender(LogRhythmAxonQueryRender):
    details: PlatformDetails = logrhythm_axon_rule_details
    mappings: LogRhythmAxonMappings = logrhythm_axon_rule_mappings
    or_token = "or"
    field_value_render = LogRhythmAxonRuleFieldValueRender(or_token=or_token)

    def finalize_query(
        self,
        prefix: str,
        query: str,
        functions: str,
        meta_info: Optional[MetaInfoContainer] = None,
        source_mapping: Optional[SourceMapping] = None,
        not_supported_functions: Optional[list] = None,
        unmapped_fields: Optional[list[str]] = None,
        *args,  # noqa: ARG002
        **kwargs,  # noqa: ARG002
    ) -> str:
        query = super().finalize_query(prefix=prefix, query=query, functions=functions)
        rule = copy.deepcopy(DEFAULT_LOGRHYTHM_AXON_RULE)
        rule["observationPipeline"]["pattern"]["operations"][0]["logObserved"]["filter"] = query
        rule["title"] = meta_info.title or _AUTOGENERATED_TEMPLATE
        rule["description"] = get_rule_description_str(
            description=meta_info.description or rule["description"] or _AUTOGENERATED_TEMPLATE,
            author=meta_info.author,
            license_=meta_info.license,
        )
        rule["observationPipeline"]["pattern"]["operations"][0]["ruleElementKey"] = meta_info.id
        rule["observationPipeline"]["metadataFields"]["threat.severity"] = _SEVERITIES_MAP.get(
            meta_info.severity, SeverityType.medium
        )
        if mitre_info := meta_info.mitre_attack:
            if tactics := mitre_info.tactics:
                rule["observationPipeline"]["metadataFields"]["threat.mitre_tactic"] = ", ".join(
                    f"{i.external_id}:{i.name}" for i in sorted(tactics, key=lambda x: x.external_id)
                )
            if techniques := mitre_info.techniques:
                rule["observationPipeline"]["metadataFields"]["threat.mitre_technique"] = ", ".join(
                    f"{i.technique_id}:{i.name}" for i in sorted(techniques, key=lambda x: x.technique_id)
                )
        if meta_info.output_table_fields:
            rule["observationPipeline"]["pattern"]["operations"][0]["logObserved"]["groupByFields"] = [
                self.mappings.map_field(field, source_mapping)[0] for field in meta_info.output_table_fields
            ]

        json_rule = json.dumps(rule, indent=4, sort_keys=False)
        json_rule = self.wrap_with_unmapped_fields(json_rule, unmapped_fields)
        return self.wrap_with_not_supported_functions(json_rule, not_supported_functions)
