"""
Uncoder IO Community Edition License
-----------------------------------------------------------------
Copyright (c) 2024 SOC Prime, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
-----------------------------------------------------------------
"""

import copy
import json
from typing import Optional

from app.translator.core.mapping import SourceMapping
from app.translator.core.mitre import MitreConfig
from app.translator.core.models.platform_details import PlatformDetails
from app.translator.core.models.query_container import MetaInfoContainer, MitreInfoContainer
from app.translator.managers import render_manager
from app.translator.platforms.base.lucene.mapping import LuceneMappings
from app.translator.platforms.elasticsearch.const import XPACK_WATCHER_RULE, xpack_watcher_details
from app.translator.platforms.elasticsearch.mapping import xpack_watcher_mappings
from app.translator.platforms.elasticsearch.renders.elasticsearch import (
    ElasticSearchFieldValue,
    ElasticSearchQueryRender,
)
from app.translator.tools.utils import get_rule_description_str

_AUTOGENERATED_TEMPLATE = "Autogenerated Elastic Watcher"


class XpackWatcherRuleFieldValue(ElasticSearchFieldValue):
    details: PlatformDetails = xpack_watcher_details


@render_manager.register
class XPackWatcherRuleRender(ElasticSearchQueryRender):
    details: PlatformDetails = xpack_watcher_details
    mappings: LuceneMappings = xpack_watcher_mappings
    or_token = "OR"
    field_value_render = XpackWatcherRuleFieldValue(or_token=or_token)
    mitre: MitreConfig = MitreConfig()

    def __create_mitre_threat(self, mitre_attack: MitreInfoContainer) -> dict:
        result = {"tactics": [], "techniques": []}

        for tactic in mitre_attack.tactics:
            result["tactics"].append({"external_id": tactic.external_id, "url": tactic.url, "tactic": tactic.name})
        for technique in mitre_attack.techniques:
            result["techniques"].append(
                {
                    "technique_id": technique.technique_id,
                    "technique": technique.name,
                    "url": technique.url,
                    "tactic": technique.tactic,
                }
            )

        return result if result["tactics"] or result["techniques"] else {}

    def finalize_query(
        self,
        prefix: str,
        query: str,
        functions: str,
        meta_info: Optional[MetaInfoContainer] = None,
        source_mapping: Optional[SourceMapping] = None,
        not_supported_functions: Optional[list] = None,
        unmapped_fields: Optional[list[str]] = None,
        *args,  # noqa: ARG002
        **kwargs,  # noqa: ARG002
    ) -> str:
        query = super().finalize_query(prefix=prefix, query=query, functions=functions)
        rule = copy.deepcopy(XPACK_WATCHER_RULE)
        mitre_attack = []
        if meta_info and meta_info.mitre_attack:
            mitre_attack.extend([technique.technique_id for technique in meta_info.mitre_attack.techniques])
            mitre_attack.extend([tactic.name for tactic in meta_info.mitre_attack.tactics])
        rule["metadata"].update(
            {
                "query": query,
                "title": meta_info.title or _AUTOGENERATED_TEMPLATE,
                "description": get_rule_description_str(
                    description=meta_info.description or _AUTOGENERATED_TEMPLATE,
                    author=meta_info.author,
                    license_=meta_info.license,
                    mitre_attack=mitre_attack,
                ),
                "tags": self.__create_mitre_threat(mitre_attack=meta_info.mitre_attack),
            }
        )
        rule["input"]["search"]["request"]["body"]["query"]["bool"]["must"][0]["query_string"]["query"] = query
        indices = source_mapping and [str(source_mapping.log_source_signature)] or []
        rule["input"]["search"]["request"]["indices"] = indices
        rule["actions"]["send_email"]["email"]["subject"] = meta_info.title or _AUTOGENERATED_TEMPLATE
        rule_str = json.dumps(rule, indent=4, sort_keys=False)
        rule_str = self.wrap_with_unmapped_fields(rule_str, unmapped_fields)
        return self.wrap_with_not_supported_functions(rule_str, not_supported_functions)
