"""
Uncoder IO Community Edition License
-----------------------------------------------------------------
Copyright (c) 2024 SOC Prime, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
-----------------------------------------------------------------
"""

import copy
import json
from typing import Optional

from app.translator.core.mapping import SourceMapping
from app.translator.core.models.platform_details import PlatformDetails
from app.translator.core.models.query_container import MetaInfoContainer
from app.translator.managers import render_manager
from app.translator.platforms.base.lucene.mapping import LuceneMappings
from app.translator.platforms.elasticsearch.const import KIBANA_RULE, KIBANA_SEARCH_SOURCE_JSON, kibana_rule_details
from app.translator.platforms.elasticsearch.mapping import kibana_rule_mappings
from app.translator.platforms.elasticsearch.renders.elasticsearch import (
    ElasticSearchFieldValue,
    ElasticSearchQueryRender,
)
from app.translator.tools.utils import get_rule_description_str

_AUTOGENERATED_TEMPLATE = "Autogenerated Elastic Kibana Saved Search"


class KibanaFieldValue(ElasticSearchFieldValue):
    details: PlatformDetails = kibana_rule_details


@render_manager.register
class KibanaRuleRender(ElasticSearchQueryRender):
    details: PlatformDetails = kibana_rule_details
    mappings: LuceneMappings = kibana_rule_mappings
    or_token = "OR"
    field_value_render = KibanaFieldValue(or_token=or_token)

    def finalize_query(
        self,
        prefix: str,
        query: str,
        functions: str,
        meta_info: Optional[MetaInfoContainer] = None,
        source_mapping: Optional[SourceMapping] = None,  # noqa: ARG002
        not_supported_functions: Optional[list] = None,
        unmapped_fields: Optional[list[str]] = None,
        *args,  # noqa: ARG002
        **kwargs,  # noqa: ARG002
    ) -> str:
        query = super().finalize_query(prefix=prefix, query=query, functions=functions)
        search_source = copy.deepcopy(KIBANA_SEARCH_SOURCE_JSON)
        search_source["query"]["query_string"]["query"] = query
        dumped_rule = json.dumps(search_source, sort_keys=False)
        rule = copy.deepcopy(KIBANA_RULE)
        rule["_source"]["kibanaSavedObjectMeta"]["searchSourceJSON"] = dumped_rule
        rule["_source"]["title"] = meta_info.title or _AUTOGENERATED_TEMPLATE
        descr = meta_info.description or rule["_source"]["description"] or _AUTOGENERATED_TEMPLATE
        mitre_attack = []
        if meta_info and meta_info.mitre_attack:
            mitre_attack.extend([technique.technique_id for technique in meta_info.mitre_attack.techniques])
            mitre_attack.extend([tactic.name for tactic in meta_info.mitre_attack.tactics])
        rule["_source"]["description"] = get_rule_description_str(
            description=descr,
            author=meta_info.author,
            rule_id=meta_info.id,
            license_=meta_info.license,
            references=meta_info.references,
            mitre_attack=mitre_attack,
        )
        rule_str = json.dumps(rule, indent=4, sort_keys=False)
        rule_str = self.wrap_with_unmapped_fields(rule_str, unmapped_fields)
        return self.wrap_with_not_supported_functions(rule_str, not_supported_functions)
