"""
Uncoder IO Community Edition License
-----------------------------------------------------------------
Copyright (c) 2023 SOC Prime, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
-----------------------------------------------------------------
"""
import re

from typing import Union

from app.translator.const import DEFAULT_VALUE_TYPE
from app.translator.platforms.logscale.const import logscale_query_details
from app.translator.platforms.logscale.functions import LogScaleFunctions, log_scale_functions
from app.translator.platforms.logscale.mapping import LogScaleMappings, logscale_mappings
from app.translator.core.mapping import SourceMapping
from app.translator.core.models.platform_details import PlatformDetails
from app.translator.core.models.parser_output import MetaInfoContainer
from app.translator.core.render import BaseQueryRender, BaseQueryFieldValue


class LogScaleFieldValue(BaseQueryFieldValue):
    details: PlatformDetails = logscale_query_details

    def apply_value(self, value: Union[str, int]) -> str:
        if isinstance(value, str) and '"' in value:
            value = re.sub(r'(?<!\\)"', r'\"', value)
        if isinstance(value, str) and '/' in value:
            value = re.sub(r'(?<!\\)/', r'\/', value)
        return value

    def apply_field_name(self, field_name: str) -> str:
        if not field_name.isalpha():
            return f'"{field_name}"'
        return field_name

    def equal_modifier(self, field: str, value: DEFAULT_VALUE_TYPE) -> str:
        if isinstance(value, list):
            return f"({self.or_token.join(self.equal_modifier(field=field, value=v) for v in value)})"
        return f'{self.apply_field_name(field_name=field)}=/{self.apply_value(value)}/i'

    def less_modifier(self, field: str, value: Union[int, str]) -> str:
        return f'{self.apply_field_name(field_name=field)}<"{self.apply_value(value)}"'

    def less_or_equal_modifier(self, field: str, value: Union[int, str]) -> str:
        return f'{self.apply_field_name(field_name=field)}<="{self.apply_value(value)}"'

    def greater_modifier(self, field: str, value: Union[int, str]) -> str:
        return f'{self.apply_field_name(field_name=field)}>"{self.apply_value(value)}"'

    def greater_or_equal_modifier(self, field: str, value: Union[int, str]) -> str:
        return f'{self.apply_field_name(field_name=field)}>="{self.apply_value(value)}"'

    def not_equal_modifier(self, field: str, value: DEFAULT_VALUE_TYPE) -> str:
        if isinstance(value, list):
            return f"({self.or_token.join([self.not_equal_modifier(field=field, value=v) for v in value])})"
        return f'{self.apply_field_name(field_name=field)}!=/{self.apply_value(value)}/i'

    def contains_modifier(self, field: str, value: DEFAULT_VALUE_TYPE) -> str:
        if isinstance(value, list):
            return f"({self.or_token.join(self.contains_modifier(field=field, value=v) for v in value)})"
        return f'{self.apply_field_name(field_name=field)}=/{self.apply_value(value)}/i'

    def endswith_modifier(self, field: str, value: DEFAULT_VALUE_TYPE) -> str:
        if isinstance(value, list):
            return f"({self.or_token.join(self.endswith_modifier(field=field, value=v) for v in value)})"
        return f'{self.apply_field_name(field_name=field)}=/{self.apply_value(value)}$/i'

    def startswith_modifier(self, field: str, value: DEFAULT_VALUE_TYPE) -> str:
        if isinstance(value, list):
            return f"({self.or_token.join(self.startswith_modifier(field=field, value=v) for v in value)})"
        return f'{self.apply_field_name(field_name=field)}=/^{self.apply_value(value)}/i'

    def regex_modifier(self, field: str, value: DEFAULT_VALUE_TYPE) -> str:
        if isinstance(value, list):
            return f"({self.or_token.join(self.regex_modifier(field=field, value=v) for v in value)})"
        return f'{self.apply_field_name(field_name=field)}=/{self.apply_value(value)}/'

    def keywords(self, field: str, value: DEFAULT_VALUE_TYPE) -> str:
        if isinstance(value, list):
            return f"({self.or_token.join(self.keywords(field=field, value=v) for v in value)})"
        return f'/{self.apply_value(value)}/i'


class LogScaleQueryRender(BaseQueryRender):
    details: PlatformDetails = logscale_query_details
    mappings: LogScaleMappings = logscale_mappings
    platform_functions: LogScaleFunctions = log_scale_functions

    or_token = "or"
    and_token = ""
    not_token = "not"

    field_value_map = LogScaleFieldValue(or_token=or_token)
    query_pattern = "{prefix} {query} {functions}"

    def __init__(self):
        super().__init__()
        self.platform_functions.manager.init_search_func_render(self)

    def wrap_with_comment(self, value: str) -> str:
        return f"/* {value} */"

    def finalize_query(self, prefix: str, query: str, functions: str, meta_info: MetaInfoContainer = None,
                       source_mapping: SourceMapping = None, not_supported_functions: list = None) -> str:
        if prefix:
            query = self.query_pattern.format(prefix=prefix, query=query, functions=functions)
        else:
            query = f'{query} {functions.lstrip()}'
        query = self.wrap_query_with_meta_info(meta_info=meta_info, query=query)
        if not_supported_functions:
            rendered_not_supported = self.render_not_supported_functions(not_supported_functions)
            return query + rendered_not_supported
        return query
