/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pulsar.broker.admin;

import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.pulsar.broker.auth.MockedPulsarServiceBaseTest;
import org.apache.pulsar.broker.service.persistent.PersistentTopic;
import org.apache.pulsar.client.admin.PulsarAdminException;
import org.apache.pulsar.client.api.Producer;
import org.apache.pulsar.common.policies.data.ClusterData;
import org.apache.pulsar.common.policies.data.TenantInfo;
import org.testng.Assert;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import java.lang.reflect.Method;
import java.util.UUID;

@Slf4j
public class TopicMessageTTLTest extends MockedPulsarServiceBaseTest {

    private final String testTenant = "my-tenant";

    private final String testNamespace = "my-namespace";

    private final String myNamespace = testTenant + "/" + testNamespace;

    private final String testTopic = "persistent://" + myNamespace + "/test-topic-message-ttl";

    @BeforeMethod
    @Override
    protected void setup() throws Exception {
        this.conf.setSystemTopicEnabled(true);
        this.conf.setTopicLevelPoliciesEnabled(true);
        this.conf.setTtlDurationDefaultInSeconds(3600);
        super.internalSetup();

        admin.clusters().createCluster("test", new ClusterData(pulsar.getWebServiceAddress()));
        TenantInfo tenantInfo = new TenantInfo(Sets.newHashSet("role1", "role2"), Sets.newHashSet("test"));
        admin.tenants().createTenant(this.testTenant, tenantInfo);
        admin.namespaces().createNamespace(testTenant + "/" + testNamespace, Sets.newHashSet("test"));
        admin.topics().createPartitionedTopic(testTopic, 2);
        Producer producer = pulsarClient.newProducer().topic(testTenant + "/" + testNamespace + "/" + "dummy-topic").create();
        producer.close();
        Thread.sleep(3000);
    }

    @AfterMethod
    @Override
    public void cleanup() throws Exception {
        super.internalCleanup();
    }

    @Test(timeOut = 20000)
    public void testDifferentLevelPolicyPriority() throws Exception {
        final String topicName = testTopic + UUID.randomUUID();
        admin.topics().createNonPartitionedTopic(topicName);
        PersistentTopic persistentTopic = (PersistentTopic) pulsar.getBrokerService().getTopicIfExists(topicName).get().get();
        Method method = PersistentTopic.class.getDeclaredMethod("getMessageTTL");
        method.setAccessible(true);

        int namespaceMessageTTL = admin.namespaces().getNamespaceMessageTTL(myNamespace);
        Assert.assertEquals(namespaceMessageTTL, 3600);
        Assert.assertEquals((int)method.invoke(persistentTopic), 3600);

        admin.namespaces().setNamespaceMessageTTL(myNamespace, 10);
        Thread.sleep(500);
        Assert.assertEquals(admin.namespaces().getNamespaceMessageTTL(myNamespace), 10);
        Assert.assertEquals((int)method.invoke(persistentTopic), 10);

        admin.namespaces().setNamespaceMessageTTL(myNamespace, 0);
        Thread.sleep(500);
        Assert.assertEquals(admin.namespaces().getNamespaceMessageTTL(myNamespace), 0);
        Assert.assertEquals((int)method.invoke(persistentTopic), 0);

        admin.namespaces().removeNamespaceMessageTTL(myNamespace);
        Thread.sleep(500);
        Assert.assertEquals(admin.namespaces().getNamespaceMessageTTL(myNamespace), 3600);
        Assert.assertEquals((int)method.invoke(persistentTopic), 3600);
    }

}
