/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pulsar.client.impl;

import lombok.Cleanup;
import org.apache.pulsar.client.api.MessageId;
import org.apache.pulsar.client.api.ProducerConsumerBase;
import org.apache.pulsar.client.api.PulsarClientException;
import org.apache.pulsar.client.api.Schema;
import org.apache.pulsar.common.api.proto.PulsarApi;
import org.apache.pulsar.common.util.FutureUtil;
import org.testng.Assert;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

public class ProducerSemaphoreTest extends ProducerConsumerBase {

    @Override
    @BeforeMethod
    public void setup() throws Exception {
        super.internalSetup();
        super.producerBaseSetup();
    }

    @Override
    @AfterMethod
    public void cleanup() throws Exception {
        super.internalCleanup();
    }

    @Test(timeOut = 30000)
    public void testProducerSemaphoreAcquireAndRelease() throws PulsarClientException, ExecutionException, InterruptedException {

        final int pendingQueueSize = 100;

        @Cleanup
        ProducerImpl<byte[]> producer = (ProducerImpl<byte[]>) pulsarClient.newProducer()
                .topic("testProducerSemaphoreAcquire")
                .maxPendingMessages(pendingQueueSize)
                .enableBatching(false)
                .create();

        final int messages = 10;
        final List<CompletableFuture<MessageId>> futures = new ArrayList<>(messages);
        for (int i = 0; i < messages; i++) {
            futures.add(producer.newMessage().value(("Semaphore-test-" + i).getBytes()).sendAsync());
        }

        FutureUtil.waitForAll(futures).get();
        Assert.assertEquals(producer.getSemaphore().availablePermits(), pendingQueueSize);
        futures.clear();

        // Simulate replicator, non batching message but `numMessagesInBatch` of message metadata > 1
        for (int i = 0; i < messages / 2; i++) {
            PulsarApi.MessageMetadata.Builder builder = PulsarApi.MessageMetadata.newBuilder();
            builder.setNumMessagesInBatch(10);
            MessageImpl<byte[]> msg = MessageImpl.create(builder, ByteBuffer.wrap(new byte[0]), Schema.BYTES);
            futures.add(producer.sendAsync(msg));
        }
        FutureUtil.waitForAll(futures).get();

        // Here must ensure that the semaphore available permits is 0
        Assert.assertEquals(producer.getSemaphore().availablePermits(), pendingQueueSize);

        // Acquire 5 and not wait the send ack call back
        for (int i = 0; i < messages / 2; i++) {
            producer.newMessage().value(("Semaphore-test-" + i).getBytes()).sendAsync();
        }

        // Here must ensure that the Semaphore a acquired 5
        Assert.assertEquals(producer.getSemaphore().availablePermits(), pendingQueueSize - messages / 2);

    }

    /**
     * We use semaphore to limit the pending send, so we must ensure that the thread of sending message never block
     * at the pending message queue. If not, the dead lock might occur. Here is the related issue to describe the
     * dead lock happens {https://github.com/apache/pulsar/issues/5585}
     */
    @Test(timeOut = 30000)
    public void testEnsureNotBlockOnThePendingQueue() throws PulsarClientException {

        final int pendingQueueSize = 10;

        @Cleanup
        ProducerImpl<byte[]> producer = (ProducerImpl<byte[]>) pulsarClient.newProducer()
                .topic("testProducerSemaphoreAcquire")
                .maxPendingMessages(pendingQueueSize)
                .enableBatching(false)
                .blockIfQueueFull(true)
                .create();

        final int messages = 20;
        final List<CompletableFuture<MessageId>> futures = new ArrayList<>(messages);

        // Simulate replicator, non batching message but `numMessagesInBatch` of message metadata > 1
        for (int i = 0; i < messages; i++) {
            PulsarApi.MessageMetadata.Builder builder = PulsarApi.MessageMetadata.newBuilder();
            builder.setNumMessagesInBatch(10);
            MessageImpl<byte[]> msg = MessageImpl.create(builder, ByteBuffer.wrap(new byte[0]), Schema.BYTES);
            futures.add(producer.sendAsync(msg));
        }

        FutureUtil.waitForAll(futures);
        futures.clear();

        for (int i = 0; i < messages; i++) {
            futures.add(producer.newMessage().value(("Semaphore-test-" + i).getBytes()).sendAsync());
        }
        Assert.assertEquals(producer.getSemaphore().availablePermits(), 0);
    }
}
