/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pulsar.tests.integration.topologies;

import static org.testng.Assert.assertEquals;

import lombok.extern.slf4j.Slf4j;
import org.apache.pulsar.tests.integration.containers.StandaloneContainer;
import org.apache.pulsar.tests.integration.docker.ContainerExecResult;
import org.testcontainers.containers.Network;
import org.testng.annotations.DataProvider;

/**
 * A test base to run tests on standalone cluster.
 *
 * <p>Ideally we should run all integration tests on both cluster mode and standalone
 * mode. However the apache ci can't really afford to do so. so we run all the integration
 * tests on cluster mode. We only run basic validation and test new features (e.g. state)
 * on standalone.
 */
@Slf4j
public abstract class PulsarStandaloneTestBase extends PulsarTestBase {

    @DataProvider(name = "StandaloneServiceUrlAndTopics")
    public static Object[][] serviceUrlAndTopics() {
        return new Object[][] {
                // plain text, persistent topic
                {
                        container.getPlainTextServiceUrl(),
                        true,
                },
                // plain text, non-persistent topic
                {
                        container.getPlainTextServiceUrl(),
                        false
                }
        };
    }

    protected static Network network;
    protected static StandaloneContainer container;

    protected void startCluster(final String pulsarImageName) throws Exception {
        network = Network.newNetwork();
        String clusterName = PulsarClusterTestBase.randomName(8);
        container = new StandaloneContainer(clusterName, pulsarImageName)
            .withNetwork(network)
            .withNetworkAliases(StandaloneContainer.NAME + "-" + clusterName)
            .withEnv("PF_stateStorageServiceUrl", "bk://localhost:4181");
        container.start();
        log.info("Pulsar cluster {} is up running:", clusterName);
        log.info("\tBinary Service Url : {}", container.getPlainTextServiceUrl());
        log.info("\tHttp Service Url : {}", container.getHttpServiceUrl());

        // add cluster to public tenant
        ContainerExecResult result = container.execCmd(
                "/pulsar/bin/pulsar-admin", "namespaces", "policies", "public/default");
        assertEquals(0, result.getExitCode());
        log.info("public/default namespace policies are {}", result.getStdout());
    }

    protected void stopCluster() throws Exception {
        container.stop();
        network.close();
    }

}
