/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
package org.elasticsearch.xpack.ml.integration;

import org.elasticsearch.action.DocWriteRequest;
import org.elasticsearch.action.admin.indices.refresh.RefreshRequest;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.xpack.core.ml.dataframe.DataFrameAnalyticsConfig;
import org.elasticsearch.xpack.core.ml.dataframe.analyses.BoostedTreeParams;
import org.elasticsearch.xpack.core.ml.dataframe.analyses.Classification;
import org.junit.After;
import org.junit.Before;

import java.util.List;
import java.util.Map;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.hasKey;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;

/**
 * This test targets the problem of missing feature importance values on the
 * house pricing
 * dataset described in https://github.com/elastic/elasticsearch/issues/88536.
 */
public class ClassificationHousePricingIT extends MlNativeDataFrameAnalyticsIntegTestCase {

    static final List<String> DATA = List.of(
        "856,Y,1,548,2,1,CollgCr,2003",
        "1262,Y,1,460,2,1,Veenker,1976",
        "920,Y,1,608,2,1,CollgCr,2001",
        "961,Y,1,642,3,1,Crawfor,1915",
        "1145,Y,1,836,3,1,NoRidge,2000",
        "796,Y,1,480,2,1,Mitchel,1993",
        "1694,Y,1,636,2,1,Somerst,2004",
        "1107,Y,1,484,2,1,NWAmes,1973",
        "1022,Y,0,468,2,1,OldTown,1931",
        "1077,Y,1,205,1,1,BrkSide,1939",
        "1040,Y,1,384,1,1,Sawyer,1965",
        "1182,Y,1,736,3,1,NridgHt,2005",
        "912,Y,1,352,1,1,Sawyer,1962",
        "1494,Y,1,840,3,1,CollgCr,2006",
        "1253,Y,1,352,1,1,NAmes,1960",
        "854,Y,0,576,2,1,BrkSide,1929",
        "1004,Y,1,480,2,1,NAmes,1970",
        "1296,Y,1,516,2,1,Sawyer,1967",
        "1114,Y,1,576,2,1,SawyerW,2004",
        "1339,Y,1,294,1,1,NAmes,1958",
        "1158,Y,1,853,3,1,NridgHt,2005",
        "1108,Y,0,280,1,1,IDOTRR,1930",
        "1795,Y,1,534,2,1,CollgCr,2002",
        "1060,Y,1,572,2,1,MeadowV,1976",
        "1060,Y,1,270,1,1,Sawyer,1968",
        "1600,Y,1,890,3,1,NridgHt,2007",
        "900,Y,1,576,2,1,NAmes,1951",
        "1704,Y,1,772,3,1,NridgHt,2007",
        "1600,Y,1,319,1,1,NAmes,1957",
        "520,N,1,240,1,1,BrkSide,1927",
        "649,N,1,250,1,1,IDOTRR,1920",
        "1228,Y,1,271,1,1,Sawyer,1966",
        "1234,Y,1,484,2,1,CollgCr,2007",
        "1700,Y,1,447,2,1,NAmes,1959",
        "1561,Y,1,556,2,1,NridgHt,2005",
        "1132,Y,1,691,3,1,NridgHt,2004",
        "1097,Y,1,672,2,1,CollgCr,1994",
        "1297,Y,1,498,2,1,NAmes,1954",
        "1057,Y,1,246,1,1,NAmes,1953",
        "1152,N,0,0,0,1,Edwards,1955",
        "1324,Y,1,440,2,1,NAmes,1965",
        "1328,Y,1,308,1,1,Timber,1959",
        "884,Y,1,504,2,1,SawyerW,1983",
        "938,Y,1,308,1,1,CollgCr,1975",
        "1150,Y,0,300,1,1,NAmes,1959",
        "1752,Y,1,576,2,1,NridgHt,2005",
        "1518,Y,1,670,2,1,Mitchel,2003",
        "1656,Y,1,826,3,1,Somerst,2006",
        "736,Y,1,0,0,1,OldTown,1920",
        "955,Y,1,386,1,1,Sawyer,1966",
        "794,Y,1,388,2,1,Gilbert,1997",
        "816,Y,1,528,2,1,BrkSide,1934",
        "816,N,1,516,2,1,IDOTRR,1963",
        "1842,Y,1,894,3,1,Veenker,1981",
        "1360,Y,1,572,2,1,NAmes,1955",
        "1425,Y,1,576,2,1,NAmes,1964",
        "983,Y,1,480,2,1,Somerst,1999",
        "860,Y,1,565,2,1,CollgCr,2004",
        "1426,Y,1,641,3,1,StoneBr,2006",
        "780,Y,1,352,1,1,CollgCr,1972",
        "1158,Y,1,576,2,1,SawyerW,2004",
        "581,N,1,288,1,1,IDOTRR,1920",
        "1370,Y,1,484,2,1,NridgHt,2006",
        "902,Y,1,480,2,1,OldTown,1921",
        "1057,Y,1,645,2,1,CollgCr,1997",
        "1143,Y,1,852,3,1,NridgHt,2004",
        "2207,Y,1,576,2,1,NAmes,1970",
        "1479,Y,1,558,2,1,CollgCr,2003",
        "747,Y,1,220,1,1,OldTown,1945",
        "1304,Y,1,667,2,1,ClearCr,1953",
        "2223,Y,1,516,2,1,NAmes,1973",
        "845,Y,1,360,2,1,Mitchel,1982",
        "885,Y,1,427,2,1,Gilbert,1998",
        "1086,Y,1,490,2,1,NAmes,1954",
        "840,N,1,379,1,1,OldTown,1915",
        "526,Y,1,297,1,1,MeadowV,1973",
        "952,Y,0,283,1,1,NAmes,1956",
        "1072,Y,1,240,1,1,BrkSide,1948",
        "1768,N,1,0,0,1,Sawyer,1968",
        "682,Y,1,440,2,1,OldTown,1910",
        "1182,Y,1,509,1,1,NAmes,1968",
        "1337,Y,1,405,2,1,Mitchel,1998",
        "1563,Y,1,758,3,1,Somerst,2007",
        "1065,Y,1,461,2,1,NAmes,1960",
        "804,Y,1,400,2,1,Gilbert,1995",
        "1301,Y,1,462,2,1,NoRidge,1991",
        "684,Y,1,400,2,1,Gilbert,2005",
        "612,Y,1,528,2,1,Somerst,2009",
        "1013,N,1,0,0,1,IDOTRR,1915",
        "990,Y,1,0,0,1,CollgCr,1994",
        "1040,Y,0,420,2,1,NAmes,1950",
        "1235,Y,1,480,2,1,NAmes,1961",
        "964,Y,1,432,2,1,Crawfor,1921",
        "1260,N,1,506,2,1,OldTown,1910",
        "905,Y,1,684,2,1,CollgCr,1997",
        "680,Y,1,420,2,1,Gilbert,1993",
        "1588,Y,1,472,2,1,CollgCr,1999",
        "960,Y,0,432,1,1,Edwards,1965",
        "835,N,1,366,1,1,Edwards,1920",
        "1225,Y,1,0,0,1,NAmes,1959",
        "1610,Y,1,480,2,1,NWAmes,1977",
        "977,Y,1,476,2,1,SawyerW,1985",
        "1535,Y,1,410,2,1,SawyerW,1979",
        "1226,Y,1,740,3,1,CollgCr,2009",
        "1226,Y,0,240,1,1,IDOTRR,1931",
        "1053,Y,1,648,2,1,Somerst,2003",
        "1047,Y,1,273,1,1,OldTown,1885",
        "789,Y,1,250,1,1,OldTown,1948",
        "997,N,0,0,0,1,IDOTRR,1919",
        "1844,Y,1,546,2,1,NWAmes,1977",
        "1216,Y,1,325,1,1,Edwards,1954",
        "774,Y,1,400,2,1,Gilbert,2000",
        "1282,Y,1,792,3,1,CollgCr,2007",
        "2259,Y,1,450,2,1,Crawfor,1953",
        "1436,Y,1,180,1,1,Crawfor,1945",
        "729,Y,1,440,2,1,Somerst,1999",
        "1092,Y,1,288,1,1,Sawyer,1962",
        "1125,Y,1,430,2,1,Edwards,2006",
        "1699,Y,1,594,3,1,SawyerW,1990",
        "728,Y,1,390,2,1,CollgCr,2005",
        "988,Y,1,540,2,1,ClearCr,1969",
        "772,Y,1,264,1,1,IDOTRR,1939",
        "1080,Y,1,288,1,1,NAmes,1958",
        "1199,Y,1,530,2,1,SawyerW,1993",
        "1586,Y,1,435,2,1,NWAmes,1979",
        "520,N,1,0,0,1,IDOTRR,1935",
        "958,Y,1,440,2,1,NPkVill,1976",
        "840,Y,1,0,0,1,OldTown,1930",
        "660,Y,1,453,2,1,NAmes,1966",
        "1053,Y,0,750,2,1,NAmes,1958",
        "1216,Y,1,487,2,1,NAmes,1966",
        "1022,Y,1,390,2,1,Gilbert,2000",
        "1327,Y,1,624,2,1,NAmes,1959",
        "1296,Y,1,471,2,1,Timber,2001",
        "1721,Y,1,440,2,1,Sawyer,1968",
        "1682,Y,1,530,2,1,NWAmes,1970",
        "1214,Y,1,318,1,1,NAmes,1967",
        "1959,Y,1,766,3,1,Mitchel,1988",
        "1004,Y,1,660,3,1,CollgCr,1999",
        "928,Y,1,470,2,1,CollgCr,1997",
        "864,Y,1,0,0,1,NAmes,1971",
        "1734,Y,1,660,2,1,CollgCr,2005",
        "910,Y,1,720,2,1,NAmes,1952",
        "1501,Y,1,577,2,1,CollgCr,1999",
        "1728,Y,1,504,2,1,Sawyer,1963",
        "970,Y,1,380,2,1,Edwards,2004",
        "875,Y,0,180,1,1,BrkSide,1931",
        "884,Y,1,434,2,1,Gilbert,2001",
        "1080,Y,1,0,0,1,SawyerW,2004",
        "896,Y,0,240,1,1,BrkSide,1936",
        "969,Y,1,440,2,1,CollgCr,1975",
        "1710,Y,1,866,3,1,NridgHt,2007",
        "1097,Y,1,495,2,1,NWAmes,1971",
        "1252,Y,1,564,2,1,ClearCr,1960",
        "1200,Y,0,312,1,1,OldTown,1923",
        "572,N,0,0,0,0,Edwards,1924",
        "1040,Y,0,625,2,1,NAmes,1950",
        "774,Y,1,680,3,1,Timber,2009",
        "991,Y,1,678,2,1,Somerst,2004",
        "1392,Y,1,576,2,1,Gilbert,2005",
        "1232,Y,1,516,2,1,Veenker,1984",
        "1572,Y,1,726,3,1,NridgHt,2003",
        "1541,Y,1,532,2,1,NridgHt,2005",
        "882,Y,1,0,0,1,OldTown,1956",
        "1149,Y,0,216,1,1,OldTown,1926",
        "808,Y,1,0,0,1,Edwards,1940",
        "1867,Y,0,303,1,1,ClearCr,1955",
        "1610,Y,1,789,3,1,NridgHt,2007",
        "840,Y,1,440,2,1,Gilbert,2004",
        "1707,Y,1,511,2,1,Timber,1981",
        "854,Y,1,660,2,1,OldTown,1941",
        "1656,Y,1,528,2,1,NAmes,1960",
        "1064,Y,1,504,2,1,StoneBr,1987",
        "1362,Y,1,504,2,1,NAmes,1961",
        "1651,Y,1,616,2,1,Timber,1986",
        "2158,Y,1,576,2,1,Edwards,1950",
        "1164,Y,1,521,2,1,ClearCr,1988",
        "1252,Y,1,451,2,1,Sawyer,1958",
        "2234,Y,1,1166,3,1,StoneBr,2008",
        "968,Y,1,480,2,1,OldTown,1923",
        "769,Y,1,440,2,1,Somerst,2000",
        "901,Y,1,216,1,1,Crawfor,1920",
        "1340,Y,1,252,1,1,Edwards,1957",
        "1362,Y,1,484,2,1,OldTown,2003",
        "936,Y,1,576,2,1,BrkSide,1908",
        "1518,Y,1,840,3,1,OldTown,1892",
        "1217,Y,1,497,2,1,Mitchel,1990",
        "808,Y,1,180,1,1,OldTown,1916",
        "1224,Y,1,528,2,1,SawyerW,1979",
        "1593,Y,1,682,2,1,StoneBr,2001",
        "1549,Y,1,440,2,1,Crawfor,1932",
        "725,Y,1,484,2,1,NAmes,1972",
        "1431,Y,1,666,2,1,CollgCr,1999",
        "970,Y,1,380,2,1,Edwards,2004",
        "864,Y,1,352,1,1,CollgCr,1972",
        "855,Y,1,440,2,1,NPkVill,1976",
        "1726,Y,1,786,3,1,Somerst,2007",
        "1360,Y,1,795,2,1,NAmes,1918",
        "929,Y,1,0,0,1,OldTown,1912",
        "1713,Y,1,856,3,1,NridgHt,2004",
        "1121,Y,1,440,2,1,Edwards,2003",
        "1279,Y,1,473,2,1,Mitchel,1977",
        "865,Y,1,398,1,1,OldTown,1924",
        "848,Y,1,420,2,1,CollgCr,2004",
        "720,Y,1,240,1,1,OldTown,1947",
        "1442,Y,1,500,2,1,Gilbert,1990",
        "1696,Y,1,349,1,1,Sawyer,1962",
        "1100,Y,1,312,1,1,NAmes,1960",
        "1180,Y,1,454,2,1,SawyerW,1988",
        "1092,Y,1,504,2,1,NAmes,1964",
        "864,N,0,0,0,1,Edwards,1925",
        "1212,Y,1,460,2,1,Edwards,2009",
        "932,Y,1,644,2,1,Somerst,2009",
        "990,Y,1,576,2,1,CollgCr,1995",
        "689,Y,1,299,1,1,CollgCr,1977",
        "1236,Y,1,447,1,1,NAmes,1957",
        "1436,Y,1,484,2,1,CollgCr,2004",
        "810,N,1,210,1,1,OldTown,1925",
        "1137,Y,1,431,2,1,Crawfor,1939",
        "1248,Y,1,438,2,1,Blmngtn,2005",
        "1498,Y,1,675,2,1,CollgCr,2006",
        "1010,Y,1,390,2,1,Gilbert,2002",
        "811,Y,1,434,2,1,NWAmes,1975",
        "864,Y,1,576,2,1,NAmes,1971",
        "2392,Y,1,968,3,1,NridgHt,2003",
        "630,Y,1,280,1,1,BrDale,1971",
        "1214,Y,1,721,3,1,NoRidge,1995",
        "483,Y,1,280,1,1,BrDale,1970",
        "912,Y,1,336,1,1,Sawyer,1967",
        "1555,Y,1,430,2,1,Blmngtn,2005",
        "1194,Y,1,312,1,1,NAmes,1959",
        "1490,Y,1,810,3,1,NoRidge,1995",
        "483,Y,1,288,1,1,BrDale,1972",
        "894,Y,1,308,1,1,CollgCr,1976",
        "860,Y,1,440,2,1,Gilbert,2002",
        "483,Y,1,264,1,1,BrDale,1971",
        "1414,Y,1,494,2,1,CollgCr,2004",
        "1014,Y,1,457,2,1,SawyerW,1993",
        "1694,Y,1,818,3,1,NridgHt,2007",
        "798,Y,0,220,1,1,Edwards,1945",
        "1566,Y,1,750,2,1,Somerst,2008",
        "866,Y,1,0,0,1,OldTown,1945",
        "889,N,0,352,1,1,OldTown,1900",
        "626,Y,1,288,1,1,SawyerW,1980",
        "1222,Y,1,463,2,1,SawyerW,1994",
        "1872,Y,1,604,2,1,NWAmes,1988",
        "908,N,0,440,1,1,OldTown,1910",
        "1375,Y,1,451,2,1,NAmes,1954",
        "840,Y,1,500,2,1,CollgCr,2003",
        "1444,Y,1,389,2,1,ClearCr,1958",
        "1306,Y,1,0,0,1,BrkSide,1940",
        "1625,Y,1,538,2,1,Crawfor,2006",
        "798,Y,1,520,2,1,SawyerW,2004",
        "1302,Y,1,309,1,1,NAmes,1964",
        "1314,Y,1,294,1,1,NAmes,1957",
        "1005,Y,1,429,2,1,Gilbert,1999",
        "864,Y,1,673,2,1,Somerst,2003",
        "1604,Y,1,660,2,1,CollgCr,2006",
        "963,Y,1,564,2,1,CollgCr,2001",
        "882,Y,0,308,1,1,OldTown,1956",
        "1382,Y,1,884,2,1,NAmes,1962",
        "1482,Y,1,868,3,1,CollgCr,2007",
        "1212,Y,1,492,2,1,Sawyer,1977",
        "926,Y,1,484,2,1,OldTown,1929",
        "764,N,0,504,2,1,OldTown,1925",
        "1422,Y,1,576,2,1,NWAmes,1981",
        "802,Y,1,413,2,1,Gilbert,1997",
        "1052,Y,1,240,1,1,SWISU,1939",
        "778,Y,1,924,1,1,IDOTRR,1940",
        "1113,Y,1,504,1,1,Edwards,1976",
        "1095,Y,1,1053,3,1,Somerst,2006",
        "1363,Y,1,439,2,1,ClearCr,1954",
        "1164,Y,1,671,3,1,NoRidge,1999",
        "1632,Y,1,338,1,1,NAmes,1958",
        "816,Y,1,264,1,1,Mitchel,1982",
        "952,N,1,672,2,1,BrkSide,1925",
        "1560,Y,1,573,2,1,Mitchel,2003",
        "864,Y,1,400,2,1,Gilbert,1951",
        "2121,Y,1,732,3,1,NridgHt,2006",
        "1156,Y,1,505,2,1,ClearCr,1977",
        "1175,Y,1,575,2,1,SawyerW,1989",
        "1262,Y,1,572,2,1,Somerst,2006",
        "1314,Y,1,626,2,1,NridgHt,2007",
        "1468,Y,1,898,3,1,Somerst,2008",
        "1575,Y,1,529,2,1,SawyerW,1992",
        "625,Y,1,528,2,1,Somerst,2006",
        "1085,Y,1,440,2,1,NAmes,1962",
        "858,Y,1,0,0,1,NAmes,1971",
        "900,Y,1,280,1,1,Sawyer,1967",
        "698,Y,0,384,1,1,BrkSide,1915",
        "1079,Y,1,685,2,1,CollgCr,2006",
        "936,N,1,0,0,1,SWISU,1912",
        "1148,Y,1,281,1,1,Edwards,1949",
        "1468,Y,1,539,2,1,NWAmes,1977",
        "1644,Y,1,418,2,1,NAmes,1953",
        "1003,Y,1,588,2,1,Mitchel,1984",
        "910,Y,0,282,1,1,IDOTRR,1950",
        "975,Y,1,576,2,1,Somerst,1997",
        "1041,Y,1,539,2,1,NWAmes,1968",
        "1152,Y,1,300,1,1,Crawfor,1950",
        "1336,Y,1,375,1,1,Crawfor,1953",
        "1210,Y,1,683,2,1,CollgCr,1998",
        "1541,Y,1,843,3,1,CollgCr,2001",
        "894,Y,1,552,2,1,CollgCr,1972",
        "1675,Y,1,870,3,1,OldTown,1880",
        "2000,Y,1,888,3,1,CollgCr,2004",
        "1122,Y,1,746,3,1,SawyerW,1990",
        "1035,Y,0,0,0,1,IDOTRR,1920",
        "861,Y,1,539,2,1,Edwards,1940",
        "1944,Y,1,708,3,1,NridgHt,2003",
        "697,Y,1,420,2,1,Gilbert,1993",
        "972,Y,1,240,1,1,NAmes,1948",
        "793,Y,1,410,2,1,OldTown,1939",
        "2036,Y,1,513,2,1,Timber,1965",
        "832,Y,1,546,2,1,OldTown,1925",
        "716,Y,1,432,2,1,Gilbert,2004",
        "1153,Y,1,484,2,1,NWAmes,1980",
        "1088,Y,1,1025,3,1,Somerst,2006",
        "1372,Y,1,656,3,1,NoRidge,1993",
        "1472,Y,1,588,2,1,NWAmes,1980",
        "1249,Y,1,840,3,1,NridgHt,2006",
        "1136,Y,1,872,3,1,NridgHt,2004",
        "1553,Y,1,576,2,1,SawyerW,1986",
        "1163,Y,1,220,1,1,OldTown,1955",
        "1898,Y,1,564,2,1,NAmes,1967",
        "803,N,0,360,2,1,IDOTRR,1941",
        "1719,Y,1,473,2,1,Veenker,1993",
        "1383,Y,1,292,1,1,NAmes,1960",
        "1445,N,0,441,2,1,BrkSide,1916",
        "596,N,1,189,1,1,IDOTRR,1920",
        "1728,Y,1,352,1,1,NAmes,1964",
        "1056,Y,1,308,1,1,NAmes,1958",
        "1629,Y,1,880,3,1,NridgHt,2003",
        "1358,Y,1,484,2,1,NridgHt,2004",
        "943,Y,1,472,2,1,Gilbert,1998",
        "1619,Y,1,529,2,1,Timber,1965",
        "1922,Y,1,676,3,1,StoneBr,2005",
        "1536,Y,1,532,2,1,CollgCr,2002",
        "1621,Y,1,440,2,1,NWAmes,1984",
        "1215,Y,0,297,1,1,NAmes,1958",
        "993,Y,1,431,2,1,Timber,2002",
        "841,Y,1,294,1,1,SawyerW,1950",
        "1040,N,0,400,2,0,NAmes,1949",
        "1684,Y,1,564,2,1,NridgHt,2005",
        "536,Y,1,336,1,1,MeadowV,1976",
        "972,Y,1,312,1,1,BrkSide,1939",
        "958,Y,1,301,1,1,NAmes,1960",
        "1478,Y,0,498,2,1,NAmes,1960",
        "764,Y,1,474,2,1,NridgHt,2003",
        "1848,Y,1,706,3,1,NridgHt,2005",
        "1869,Y,1,617,2,1,NridgHt,2007",
        "1453,Y,1,445,2,1,ClearCr,1986",
        "616,N,1,200,1,1,Edwards,1941",
        "720,Y,1,484,2,1,OldTown,1928",
        "1192,Y,1,240,1,1,SWISU,1940",
        "1200,Y,1,521,2,1,CollgCr,1995",
        "1167,Y,1,400,2,1,Gilbert,1992",
        "1142,Y,1,528,2,1,MeadowV,1976",
        "1352,Y,0,288,1,1,ClearCr,1958",
        "1086,Y,1,592,2,1,NoRidge,1998",
        "912,Y,1,470,2,1,Mitchel,1978",
        "988,Y,1,240,1,1,BrkSide,1940",
        "495,Y,1,672,2,1,Edwards,2003",
        "483,Y,1,264,1,1,BrDale,1972",
        "790,Y,1,566,2,1,NWAmes,1976",
        "672,Y,0,468,1,1,IDOTRR,1920",
        "1394,Y,1,514,2,1,NAmes,1963",
        "1431,Y,1,296,1,1,NAmes,1962",
        "1268,Y,1,244,1,1,NAmes,1954",
        "1287,Y,1,576,2,1,NAmes,1959",
        "953,Y,1,460,2,1,Gilbert,2000",
        "1120,Y,1,680,2,1,ClearCr,1959",
        "752,Y,1,264,1,1,SawyerW,1984",
        "1319,Y,1,270,1,1,NAmes,1953",
        "847,Y,1,434,2,1,CollgCr,2003",
        "904,N,0,0,0,1,Edwards,1922",
        "914,Y,1,576,2,1,CollgCr,1996",
        "1580,Y,1,610,2,1,Somerst,2004",
        "1856,Y,1,834,3,1,StoneBr,2010",
        "1007,Y,1,463,2,1,Gilbert,2000",
        "1026,Y,1,308,1,1,SWISU,1924",
        "1301,Y,1,572,2,1,Somerst,2006",
        "939,Y,1,639,2,1,CollgCr,2006",
        "784,N,0,360,2,1,SawyerW,1928",
        "1079,Y,1,501,2,1,ClearCr,1992",
        "1269,Y,1,430,2,1,Blmngtn,2004",
        "658,Y,1,0,0,1,Edwards,1910",
        "1125,Y,1,352,1,1,Edwards,1976",
        "1479,Y,1,577,2,1,CollgCr,1999",
        "1742,Y,1,846,3,1,NridgHt,2007",
        "961,Y,1,384,1,1,Edwards,1900",
        "804,Y,1,560,2,1,Mitchel,2001",
        "882,Y,1,294,1,1,NAmes,1959",
        "788,Y,0,0,0,1,BrkSide,1941",
        "735,Y,0,240,1,1,OldTown,1940",
        "1144,Y,1,596,1,1,Edwards,1956",
        "894,Y,1,600,2,1,CollgCr,1972",
        "876,Y,1,264,1,1,NAmes,1962",
        "1077,Y,0,338,1,1,IDOTRR,1920",
        "1112,Y,1,438,2,1,Somerst,2006",
        "1288,Y,1,500,2,1,Veenker,1996",
        "1310,Y,1,400,2,1,CollgCr,2005",
        "672,Y,1,240,1,1,Sawyer,1940",
        "1165,Y,1,420,2,1,NoRidge,1998",
        "806,Y,1,373,2,1,Gilbert,1995",
        "1620,Y,1,490,2,1,Sawyer,1976",
        "1166,Y,0,240,1,1,SWISU,1936",
        "840,Y,1,308,1,1,Crawfor,1915",
        "1071,Y,1,947,3,1,NridgHt,2006",
        "1050,Y,1,836,3,1,Somerst,2007",
        "1276,Y,0,350,1,1,Edwards,1958",
        "1056,N,1,572,2,1,Gilbert,1955",
        "1478,Y,1,484,2,1,Somerst,2009",
        "1028,Y,0,360,2,1,OldTown,1927",
        "1080,Y,1,678,3,1,SawyerW,1993",
        "1340,Y,1,396,2,1,Gilbert,2007",
        "672,Y,1,440,2,1,Sawyer,1978",
        "1370,N,0,864,3,1,Crawfor,1918",
        "756,N,0,240,1,1,Edwards,1940",
        "1056,Y,1,304,1,1,NAmes,1968",
        "1344,Y,1,784,4,1,Mitchel,1997",
        "1602,Y,1,529,2,1,NWAmes,1977",
        "988,Y,0,520,2,1,Mitchel,1954",
        "1470,Y,1,696,3,1,NoRidge,1998",
        "1196,Y,0,297,1,1,NAmes,1956",
        "707,Y,1,240,1,1,OldTown,1946",
        "1644,Y,1,569,2,1,SawyerW,1989",
        "907,Y,1,352,1,1,NAmes,1957",
        "1208,Y,1,628,2,1,CollgCr,2007",
        "1412,Y,1,576,2,1,Timber,1988",
        "483,Y,1,264,1,1,BrDale,1971",
        "1088,Y,1,0,0,1,OldTown,1920",
        "765,Y,1,440,2,1,BrDale,1971",
        "926,Y,1,470,2,1,Gilbert,1997",
        "630,Y,1,0,0,1,MeadowV,1972",
        "827,Y,1,550,2,1,CollgCr,1996",
        "734,Y,0,440,2,1,OldTown,1920",
        "904,Y,1,180,1,1,BrkSide,1926",
        "694,N,1,352,1,1,Crawfor,1913",
        "684,Y,1,528,2,1,Edwards,1920",
        "2402,Y,1,672,3,1,NridgHt,2008",
        "1440,N,0,0,0,1,Edwards,1955",
        "1128,Y,1,360,2,1,BrkSide,1930",
        "1258,Y,1,648,3,1,Blmngtn,2006",
        "933,Y,1,493,2,1,CollgCr,1994",
        "1689,Y,1,480,2,1,Edwards,1956",
        "1888,Y,1,578,2,1,NAmes,1966",
        "956,Y,1,431,2,1,Gilbert,1998",
        "780,Y,1,198,1,1,IDOTRR,1937",
        "679,Y,1,308,1,1,OldTown,1948",
        "813,Y,0,270,1,1,OldTown,1930",
        "1533,Y,1,576,2,1,ClearCr,1975",
        "888,Y,1,422,2,1,Timber,1996",
        "786,Y,1,676,2,1,Somerst,2008",
        "1728,Y,1,560,2,1,Mitchel,1976",
        "1242,Y,1,528,2,1,NWAmes,1973",
        "624,N,1,513,3,1,OldTown,1916",
        "1663,Y,1,529,2,1,ClearCr,1954",
        "833,Y,1,228,1,1,OldTown,1925",
        "979,Y,1,352,1,1,BrkSide,1950",
        "832,Y,1,552,2,1,Somerst,2009",
        "575,Y,1,576,2,1,SWISU,1936",
        "864,Y,1,360,1,1,Sawyer,1965",
        "849,Y,0,240,1,1,Crawfor,1934",
        "1040,Y,1,0,0,1,CollgCr,1978",
        "1414,Y,1,398,2,1,Blmngtn,2004",
        "1277,Y,1,526,2,1,NAmes,1970",
        "888,Y,1,312,1,1,NAmes,1942",
        "1634,Y,1,866,3,1,NridgHt,2006",
        "832,Y,1,506,2,1,SawyerW,1993",
        "1502,Y,1,528,2,1,StoneBr,1985",
        "1161,Y,1,534,2,1,NWAmes,1977",
        "1072,Y,1,525,2,1,Edwards,2005",
        "1976,Y,1,908,3,1,NridgHt,2006",
        "1652,Y,1,499,2,1,StoneBr,2000",
        "970,Y,1,624,2,1,Sawyer,1963",
        "1493,Y,1,508,2,1,CollgCr,1997",
        "2069,Y,1,694,3,1,NridgHt,2006",
        "1718,Y,1,826,3,1,CollgCr,2007",
        "1131,Y,1,672,2,1,OldTown,1937",
        "1850,Y,1,772,3,1,NridgHt,2004",
        "1792,Y,1,874,3,1,NridgHt,2003",
        "916,Y,1,164,1,1,OldTown,1915",
        "1216,Y,1,402,2,1,Mitchel,1998",
        "999,Y,1,264,1,1,Sawyer,1962",
        "1113,Y,1,264,1,1,NAmes,1950",
        "1073,Y,1,515,2,1,NAmes,1965",
        "1484,Y,1,487,2,1,NWAmes,1971",
        "1766,N,0,520,2,1,OldTown,1900",
        "630,Y,1,286,1,1,MeadowV,1970",
        "616,Y,1,336,1,1,MeadowV,1976",
        "958,Y,0,240,1,1,NAmes,1941",
        "728,Y,1,429,2,1,Gilbert,2006",
        "1269,Y,0,308,1,1,NAmes,1960",
        "886,Y,0,273,1,1,OldTown,1938",
        "720,N,0,0,0,1,IDOTRR,1920",
        "3228,Y,1,546,2,1,NoRidge,1992",
        "1133,Y,1,240,1,1,BrkSide,1925",
        "899,Y,1,288,1,1,Sawyer,1967",
        "912,Y,1,297,1,1,NAmes,1958",
        "672,Y,1,264,1,1,BrDale,1973",
        "866,Y,1,603,2,1,Somerst,2005",
        "1214,Y,1,461,2,1,Edwards,1965",
        "1801,Y,1,484,2,1,Crawfor,1959",
        "855,Y,1,440,2,1,NPkVill,1974",
        "960,Y,1,400,2,1,OldTown,1952",
        "1065,Y,1,471,2,1,SawyerW,1993",
        "1218,Y,1,676,2,1,Somerst,2009",
        "689,Y,1,360,2,1,OldTown,1928",
        "1041,Y,1,270,1,1,NAmes,1959",
        "1363,Y,1,288,1,1,NAmes,1951",
        "1368,Y,1,474,2,1,NridgHt,2005",
        "864,Y,1,624,2,1,NAmes,1958",
        "1080,Y,1,484,2,1,Mitchel,1983",
        "789,N,1,200,1,0,Crawfor,1926",
        "2020,Y,1,900,3,1,NridgHt,2009",
        "1378,Y,1,583,2,1,NWAmes,1972",
        "1277,Y,1,889,3,1,NoRidge,1996",
        "882,Y,1,546,2,1,CollgCr,1998",
        "1276,Y,1,282,1,1,Crawfor,1926",
        "694,N,0,0,0,1,OldTown,1900",
        "1244,Y,0,336,1,1,NAmes,1957",
        "1004,Y,1,420,2,1,BrkSide,1947",
        "3138,Y,1,884,3,1,Edwards,2007",
        "1383,Y,1,834,3,1,NoRidge,1996",
        "1266,Y,1,453,2,1,Somerst,2005",
        "928,Y,1,252,1,1,NAmes,1956",
        "1476,Y,1,858,3,1,NridgHt,2008",
        "605,Y,1,0,0,1,Edwards,1920",
        "2515,Y,1,484,2,1,Crawfor,1957",
        "1509,Y,1,600,2,1,Timber,1988",
        "751,Y,1,502,2,1,BrkSide,1920",
        "827,Y,1,392,1,1,NAmes,1955",
        "334,N,0,0,0,1,BrkSide,1946",
        "707,Y,1,403,2,1,Gilbert,2004",
        "820,Y,1,0,0,1,Edwards,1910",
        "880,Y,1,527,2,1,CollgCr,1998",
        "864,Y,1,576,2,1,NAmes,1972",
        "1159,Y,1,336,1,1,Sawyer,1968",
        "1601,Y,1,670,2,1,CollgCr,2001",
        "1838,Y,1,765,3,1,Timber,2006",
        "997,Y,1,648,3,1,NoRidge,2000",
        "1680,Y,1,583,2,1,NWAmes,1998",
        "767,Y,1,367,1,1,NAmes,1998",
        "664,Y,1,426,2,1,Gilbert,2006",
        "1377,Y,1,786,3,1,NWAmes,1988",
        "915,Y,0,440,2,1,BrkSide,1923",
        "768,Y,1,624,2,1,Mitchel,1970",
        "825,Y,1,720,2,1,OldTown,1955",
        "912,Y,1,615,2,1,Somerst,2003",
        "1069,Y,1,440,2,1,NPkVill,1977",
        "928,Y,0,288,1,1,OldTown,1957",
        "1717,Y,1,908,3,1,NridgHt,2006",
        "1126,Y,1,520,2,1,Edwards,1949",
        "1006,Y,1,871,3,1,NridgHt,2003",
        "1048,Y,0,280,1,1,BrkSide,1922",
        "1092,Y,0,299,1,1,NAmes,1957",
        "897,Y,1,570,1,1,IDOTRR,1920",
        "729,Y,1,406,2,1,Gilbert,1996",
        "1557,Y,1,420,2,1,Blmngtn,2003",
        "1392,Y,1,528,2,1,Sawyer,1957",
        "1389,Y,1,418,2,1,Mitchel,1974",
        "996,Y,1,0,0,1,Edwards,1940",
        "1163,Y,1,396,2,1,Edwards,1918",
        "1166,Y,1,590,2,1,NoRidge,1992",
        "841,N,0,216,1,1,SWISU,1915",
        "1134,Y,1,656,3,1,StoneBr,2005",
        "1535,Y,1,532,2,1,NridgHt,2004",
        "1496,Y,1,612,2,1,SawyerW,1983",
        "943,Y,1,600,2,1,NAmes,1979",
        "1728,Y,1,576,2,1,NAmes,1965",
        "864,Y,1,288,1,1,NAmes,1959",
        "846,Y,1,650,2,1,Timber,2009",
        "774,Y,1,400,2,1,Gilbert,2000",
        "576,Y,1,288,1,1,NAmes,1971",
        "832,Y,1,336,1,1,NAmes,1947",
        "877,Y,1,216,1,1,SWISU,1928",
        "1320,Y,1,564,2,1,Sawyer,1966",
        "703,Y,1,540,2,1,Somerst,2007",
        "1050,N,0,352,1,1,OldTown,1954",
        "1429,Y,1,572,2,1,NAmes,1960",
        "2042,Y,1,1390,3,1,NridgHt,2008",
        "816,Y,1,0,0,1,Edwards,1990",
        "1521,Y,1,880,2,1,OldTown,1893",
        "989,Y,1,240,1,1,BrkSide,1935",
        "2028,Y,1,880,3,1,Timber,2005",
        "838,Y,1,275,1,1,BrkSide,1918",
        "860,Y,1,528,2,1,Sawyer,1982",
        "1473,Y,1,452,1,1,ClearCr,1968",
        "779,Y,0,308,1,1,BrkSide,1930",
        "770,Y,1,520,2,1,CollgCr,2004",
        "1728,Y,1,842,3,1,NridgHt,2008",
        "816,Y,1,816,2,1,Mitchel,1982",
        "848,Y,1,420,2,1,CollgCr,2003",
        "924,Y,1,280,1,1,CollgCr,1975",
        "1826,Y,1,758,3,1,StoneBr,2005",
        "684,N,0,216,1,1,OldTown,1910",
        "1402,Y,1,648,3,1,Blmngtn,2006",
        "1647,Y,1,621,2,1,Crawfor,1977",
        "716,Y,1,452,2,1,Blueste,1980",
        "1058,Y,1,736,2,1,NridgHt,2005",
        "780,Y,1,544,1,1,IDOTRR,1937",
        "927,Y,1,506,2,1,SawyerW,1992",
        "600,Y,1,480,2,1,Somerst,2004",
        "1494,Y,1,530,2,1,CollgCr,2002",
        "1186,Y,1,486,2,1,NAmes,1965",
        "1040,Y,1,576,2,1,CollgCr,1996",
        "1112,Y,1,230,1,1,Edwards,1948",
        "1940,Y,1,380,2,1,Crawfor,1934",
        "1029,Y,1,261,1,1,Sawyer,1961",
        "1476,Y,1,736,3,1,CollgCr,2000",
        "1032,Y,1,564,2,1,NWAmes,1978",
        "1299,Y,1,531,2,1,CollgCr,2001",
        "1120,Y,1,0,0,1,Mitchel,2007",
        "630,Y,1,0,0,1,MeadowV,1972",
        "1054,Y,1,480,2,1,NAmes,1963",
        "807,Y,1,393,2,1,Gilbert,2002",
        "832,Y,1,528,2,1,NAmes,1954",
        "1828,Y,1,774,3,1,NridgHt,2007",
        "1482,Y,1,749,3,1,Timber,2003",
        "864,N,0,0,0,1,Edwards,1914",
        "1548,Y,1,624,2,1,NWAmes,1974",
        "980,Y,1,484,2,1,Sawyer,1977",
        "756,Y,1,440,2,1,Somerst,2000",
        "1012,Y,1,484,2,1,NWAmes,1972",
        "1116,Y,1,440,2,1,NAmes,1962",
        "1422,Y,1,286,1,1,NAmes,1960",
        "1520,Y,1,364,1,1,NAmes,1955",
        "1040,Y,1,504,2,1,NAmes,1969",
        "1350,Y,1,520,2,1,NAmes,1964",
        "1089,Y,0,240,1,1,OldTown,1880",
        "1554,Y,1,627,2,1,NridgHt,2006",
        "1411,Y,1,544,2,1,NWAmes,1977",
        "1056,Y,1,260,1,1,NAmes,1954",
        "1056,Y,1,576,2,1,OldTown,1980",
        "1440,Y,0,0,0,1,SWISU,1914",
        "800,N,0,0,0,0,BrkSide,1936",
        "811,Y,0,256,1,1,OldTown,1954",
        "796,Y,0,0,0,1,Edwards,1910",
        "1567,Y,1,648,3,1,Blmngtn,2006",
        "1518,Y,1,588,2,1,NridgHt,2003",
        "1057,Y,1,650,2,1,Somerst,2001",
        "2000,Y,1,538,2,1,NAmes,1972",
        "780,Y,1,462,2,1,NWAmes,1969",
        "1766,Y,1,478,3,1,Somerst,2009",
        "981,Y,1,576,2,1,NAmes,1971",
        "1048,Y,1,420,2,1,NAmes,1950",
        "1094,Y,1,495,2,1,Edwards,1953",
        "1051,Y,1,442,2,1,NAmes,1966",
        "630,Y,1,0,0,1,MeadowV,1970",
        "822,Y,1,562,2,1,Somerst,2007",
        "755,Y,1,296,1,1,Edwards,1940",
        "909,Y,1,512,2,1,CollgCr,1996",
        "756,Y,1,216,1,1,IDOTRR,1906",
        "2113,Y,1,839,3,1,NoRidge,1995",
        "525,Y,1,264,1,1,BrDale,1971",
        "1053,Y,1,312,1,1,NAmes,1959",
        "851,Y,1,270,1,1,Crawfor,1931",
        "912,Y,1,330,1,1,NAmes,1948",
        "1486,Y,1,480,2,1,Edwards,1964",
        "1142,Y,1,550,2,1,NWAmes,1976",
        "1686,Y,1,711,3,1,NoRidge,1994",
        "1392,Y,1,576,2,1,NAmes,1968",
        "1181,Y,1,588,2,1,Edwards,1972",
        "2097,Y,1,1134,3,1,Somerst,2005",
        "1454,Y,1,504,2,1,Gilbert,2000",
        "1465,Y,1,596,2,1,NAmes,1965",
        "1679,Y,1,575,2,1,SawyerW,1994",
        "1437,Y,1,576,2,1,Sawyer,1956",
        "1180,Y,1,252,1,1,Crawfor,1922",
        "738,Y,1,540,2,1,CollgCr,2005",
        "697,N,1,300,1,1,Edwards,1925",
        "1208,Y,1,546,2,1,Veenker,1977",
        "1839,Y,1,416,2,1,Crawfor,1957",
        "1136,Y,1,384,1,1,NAmes,1965",
        "855,Y,1,440,2,1,NPkVill,1978",
        "1095,N,1,779,3,1,OldTown,1900",
        "792,Y,1,240,1,1,OldTown,1924",
        "2046,Y,1,834,3,1,StoneBr,2008",
        "988,Y,1,572,2,1,Sawyer,1961",
        "923,Y,1,264,1,1,SawyerW,1980",
        "848,Y,1,281,1,1,SWISU,1932",
        "1291,Y,1,431,2,1,ClearCr,1996",
        "1668,Y,1,702,3,1,CollgCr,2002",
        "1195,Y,1,486,2,1,NoRidge,1998",
        "1190,Y,1,577,2,1,StoneBr,1984",
        "874,Y,1,578,2,1,Somerst,2007",
        "551,Y,1,480,2,1,Somerst,2004",
        "1419,Y,1,567,2,1,StoneBr,2007",
        "1362,Y,1,460,2,1,NridgHt,2005",
        "848,Y,1,420,2,1,CollgCr,2004",
        "2444,Y,1,832,3,1,NoRidge,1994",
        "1238,Y,1,628,2,1,Timber,1989",
        "1073,Y,1,326,1,1,OldTown,1921",
        "1067,Y,1,576,2,1,BrkSide,1936",
        "1137,Y,1,551,2,1,Timber,1987",
        "616,Y,1,205,1,1,BrkSide,1921",
        "1148,Y,0,308,1,1,Edwards,1952",
        "894,Y,1,336,1,1,Sawyer,1965",
        "1391,Y,1,530,2,1,Somerst,2004",
        "1800,Y,1,765,3,1,Timber,2002",
        "1164,Y,1,528,2,1,NWAmes,1969",
        "1264,Y,1,666,3,1,StoneBr,2006",
        "1032,Y,1,672,2,1,OldTown,1900",
        "1484,Y,1,606,2,1,CollgCr,2004",
        "372,N,1,0,0,1,IDOTRR,1930",
        "1824,Y,1,739,2,1,ClearCr,1971",
        "1324,Y,1,550,2,1,NridgHt,2006",
        "728,Y,1,400,2,1,Gilbert,2007",
        "904,Y,1,408,1,1,Sawyer,1966",
        "729,Y,1,0,0,1,BrkSide,1935",
        "859,Y,1,384,1,1,IDOTRR,1900",
        "1228,Y,1,472,2,1,StoneBr,1988",
        "960,N,1,576,2,1,BrkSide,1970",
        "725,Y,1,475,2,1,Sawyer,1976",
        "1350,Y,1,478,2,1,NWAmes,1974",
        "1576,Y,1,704,2,1,OldTown,1890",
        "1178,Y,1,439,2,1,NWAmes,1973",
        "1325,Y,1,983,3,1,NoRidge,1993",
        "971,Y,1,300,1,1,CollgCr,1969",
        "1742,Y,1,564,2,1,StoneBr,1985",
        "848,Y,1,420,2,1,CollgCr,2004",
        "864,Y,1,463,2,1,NAmes,1970",
        "997,Y,1,548,1,1,Edwards,1954",
        "1698,Y,1,768,3,1,Edwards,2007",
        "864,Y,1,660,2,1,NAmes,1970",
        "1680,Y,1,540,2,1,Crawfor,1988",
        "1232,Y,1,632,2,1,CollgCr,2007",
        "1776,Y,1,888,3,1,NAmes,1958",
        "848,Y,0,539,2,1,IDOTRR,1925",
        "1616,Y,1,608,2,1,StoneBr,1995",
        "1146,Y,1,438,2,1,Timber,2003",
        "1153,Y,1,541,2,1,CollgCr,1998",
        "1144,Y,1,264,1,1,Sawyer,1961",
        "948,Y,1,300,1,1,Sawyer,1968",
        "880,N,1,320,2,1,OldTown,1914",
        "1040,N,0,400,2,1,NAmes,1950",
        "901,Y,1,800,3,1,Gilbert,2005",
        "1200,Y,1,0,0,1,Edwards,1987",
        "864,Y,1,572,2,1,CollgCr,2004",
        "768,Y,1,360,2,1,OldTown,1910",
        "912,Y,1,288,1,1,Sawyer,1961",
        "1349,Y,1,539,2,1,SawyerW,2000",
        "1464,Y,1,480,2,1,Sawyer,1963",
        "1337,Y,1,462,2,1,StoneBr,1993",
        "1175,Y,1,831,2,1,NWAmes,1976",
        "980,Y,1,554,2,1,Gilbert,2000",
        "1320,N,1,864,4,1,OldTown,1880",
        "1720,Y,1,527,2,1,NoRidge,1996",
        "1088,N,0,240,1,0,Edwards,1945",
        "792,Y,1,0,0,1,OldTown,1910",
        "660,Y,1,400,2,1,Gilbert,2003",
        "1494,Y,1,576,2,1,CollgCr,1997",
        "1038,Y,1,878,3,1,NridgHt,2005",
        "1026,Y,1,440,2,1,NAmes,1969",
        "742,Y,1,440,2,1,Somerst,1999",
        "866,Y,1,578,2,1,CollgCr,2007",
        "672,Y,1,440,2,1,Sawyer,1978",
        "757,Y,1,440,2,1,Somerst,1999",
        "1328,Y,1,752,3,1,NoRidge,1995",
        "864,Y,1,300,1,1,NAmes,1959",
        "1301,Y,1,440,2,1,BrkSide,1924",
        "764,Y,1,614,2,1,Somerst,2009",
        "1268,Y,1,856,3,1,NoRidge,1999",
        "1494,Y,1,481,2,1,Veenker,1995",
        "1506,Y,1,592,2,1,Somerst,2008",
        "980,Y,1,496,2,1,NWAmes,1988",
        "983,Y,1,423,1,1,Mitchel,1940",
        "1836,Y,1,484,2,1,CollgCr,2004",
        "1690,Y,1,841,3,1,StoneBr,2003",
        "858,Y,1,576,2,1,Sawyer,1982",
        "1220,Y,0,396,2,1,Edwards,1951",
        "1117,Y,1,672,2,1,Edwards,1976",
        "912,Y,0,275,1,1,NAmes,1958",
        "1973,Y,1,895,3,1,NridgHt,2006",
        "1204,Y,1,412,2,1,Mitchel,1998",
        "1614,Y,1,865,3,1,CollgCr,2005",
        "894,Y,1,440,2,1,Sawyer,1974",
        "2020,Y,1,630,2,1,SawyerW,1977",
        "1004,Y,1,504,2,1,Mitchel,1977",
        "1253,Y,1,402,2,1,Gilbert,1995",
        "810,Y,1,484,2,1,SawyerW,1992",
        "1430,Y,1,605,2,1,CollgCr,2001",
        "1110,Y,1,602,2,1,Mitchel,1978",
        "742,Y,1,0,0,1,OldTown,1914",
        "1342,Y,1,457,2,1,NWAmes,1967",
        "966,Y,1,416,1,1,OldTown,1915",
        "956,Y,1,618,2,1,SawyerW,2004",
        "901,Y,1,281,1,1,OldTown,1954",
        "976,Y,1,444,2,1,ClearCr,1966",
        "1145,Y,1,397,2,1,Blmngtn,2005",
        "1062,Y,1,539,2,1,Mitchel,1976",
        "1127,Y,1,455,2,1,NoRidge,1996",
        "1496,Y,1,474,2,1,Somerst,2007",
        "1086,Y,1,409,2,1,Gilbert,1994",
        "888,Y,1,476,2,1,SawyerW,1980",
        "1285,Y,1,528,2,1,Sawyer,1977",
        "773,Y,1,240,1,1,NAmes,1953",
        "1966,Y,1,820,3,1,NridgHt,2008",
        "981,Y,1,240,1,1,SWISU,1937",
        "616,Y,1,603,2,1,ClearCr,1997",
        "1196,Y,0,440,2,1,IDOTRR,1916",
        "728,Y,1,410,2,1,CollgCr,2005",
        "1734,Y,1,1020,3,1,NridgHt,2008",
        "1128,Y,1,286,1,1,NAmes,1954",
        "1428,Y,1,554,2,1,Somerst,2008",
        "980,Y,1,384,1,1,NAmes,1967",
        "1072,Y,1,528,2,1,BrkSide,1923",
        "1086,Y,1,484,2,1,NAmes,1966",
        "1075,N,0,360,2,1,OldTown,1898",
        "1309,Y,1,484,2,1,NWAmes,1974",
        "848,Y,1,420,2,1,CollgCr,2004",
        "1044,N,0,504,2,1,IDOTRR,1952",
        "1442,Y,1,301,1,1,NAmes,1958",
        "686,Y,1,280,1,1,Edwards,1918",
        "1661,Y,1,598,2,1,CollgCr,1998",
        "1008,Y,1,275,1,1,NAmes,1954",
        "1689,Y,1,857,3,1,Mitchel,2002",
        "1052,Y,1,440,2,1,ClearCr,1971",
        "1358,Y,1,484,2,1,NridgHt,2009",
        "798,Y,1,595,2,1,CollgCr,2003",
        "936,N,1,576,2,1,OldTown,1953",
        "847,Y,1,433,2,1,Gilbert,2003",
        "944,Y,1,240,1,1,SWISU,1940",
        "1489,Y,1,776,2,1,Somerst,2006",
        "2084,Y,1,1220,3,1,NridgHt,2007",
        "784,Y,1,0,0,1,BrkSide,1924",
        "1434,Y,1,527,2,1,SawyerW,2001",
        "1160,Y,1,538,2,1,ClearCr,1967",
        "520,Y,1,480,2,1,Somerst,2005",
        "1392,Y,0,458,2,1,NAmes,1957",
        "520,Y,1,480,2,1,Somerst,2005",
        "941,Y,1,613,2,1,CollgCr,2003",
        "1516,Y,1,472,2,1,NAmes,1964",
        "1144,Y,1,456,2,1,NAmes,1961",
        "1067,Y,1,436,2,1,Sawyer,1950",
        "1559,Y,1,812,2,1,OldTown,1948",
        "483,Y,1,264,1,1,BrDale,1973",
        "1099,Y,1,352,1,1,CollgCr,1995",
        "768,Y,1,240,1,1,Edwards,1946",
        "810,N,1,400,1,1,SWISU,1925",
        "958,Y,1,686,2,1,OldTown,1904",
        "1165,Y,1,490,2,1,NAmes,1966",
        "1800,N,1,0,0,1,NAmes,1961",
        "876,Y,1,720,3,1,OldTown,1915",
        "1701,Y,1,611,2,1,Sawyer,1975",
        "1006,Y,1,425,2,1,SawyerW,1993",
        "864,Y,1,338,1,1,CollgCr,1972",
        "1307,Y,1,360,2,1,ClearCr,1908",
        "1094,Y,1,512,2,1,Veenker,1976",
        "848,Y,1,420,2,1,CollgCr,2003",
        "1456,Y,1,400,2,1,Blmngtn,2003",
        "918,Y,1,240,1,1,Crawfor,1941",
        "1445,Y,1,645,2,1,NAmes,1964",
        "1779,Y,1,454,2,1,Sawyer,1955",
        "1040,Y,1,260,1,1,Sawyer,1962",
        "1026,Y,1,576,2,1,CollgCr,1978",
        "702,Y,1,343,2,1,Gilbert,1994",
        "1370,Y,1,479,2,1,NWAmes,1976",
        "1512,Y,1,619,2,1,NWAmes,1968",
        "912,Y,1,216,1,1,Crawfor,1918",
        "1039,Y,1,504,2,1,Sawyer,1965",
        "1097,Y,1,480,2,1,SawyerW,1984",
        "1148,Y,1,672,1,1,NAmes,1959",
        "1372,Y,1,529,2,1,Somerst,2007",
        "1002,Y,1,902,2,1,NAmes,1970",
        "1646,Y,1,870,3,1,Timber,2006",
        "1120,Y,1,544,2,1,Sawyer,1961",
        "1547,Y,1,672,2,1,Gilbert,1948",
        "1062,Y,1,574,2,1,SawyerW,1993",
        "894,N,1,308,1,1,NAmes,1962",
        "804,Y,1,523,2,1,CollgCr,1998",
        "910,Y,1,414,2,1,NAmes,1953",
        "1036,Y,1,288,1,1,NAmes,1949",
        "676,Y,1,200,1,1,OldTown,1941",
        "1184,Y,1,550,2,1,Somerst,2007",
        "1040,Y,1,648,2,1,Mitchel,1963",
        "1462,Y,1,738,3,1,NridgHt,2004",
        "1155,Y,1,576,2,1,Sawyer,1961",
        "864,Y,1,336,1,1,CollgCr,1978",
        "1090,Y,1,450,2,1,Edwards,2005",
        "1187,Y,1,400,2,1,Timber,1990",
        "808,Y,1,389,2,1,Gilbert,1992",
        "954,N,1,440,1,1,SWISU,1912",
        "892,Y,1,288,1,1,NAmes,1967",
        "1709,Y,1,506,2,1,Somerst,1999",
        "1712,Y,1,588,2,1,NAmes,1959",
        "872,Y,1,300,1,1,Edwards,1955",
        "2217,Y,1,621,2,1,NAmes,1970",
        "1505,Y,1,505,2,1,NAmes,1953",
        "672,Y,1,576,2,1,NAmes,1949",
        "918,Y,1,440,2,1,Sawyer,1978",
        "1068,Y,1,264,1,1,Sawyer,1963",
        "1383,Y,1,354,1,1,Sawyer,1954",
        "1535,Y,1,400,2,1,SawyerW,1979",
        "983,Y,1,483,2,1,NAmes,1963",
        "951,N,1,327,1,1,IDOTRR,1936",
        "1120,Y,1,528,2,1,SawyerW,1979",
        "2364,Y,1,820,3,1,NridgHt,2009",
        "1236,Y,1,288,1,1,Sawyer,1961",
        "858,Y,1,684,1,1,NAmes,1971",
        "1306,Y,1,756,1,1,NAmes,1957",
        "807,Y,1,393,2,1,Gilbert,2003",
        "1670,Y,1,690,3,1,Gilbert,2006",
        "902,Y,1,288,1,1,Sawyer,1967",
        "1063,Y,1,280,1,1,NAmes,1954",
        "1636,Y,1,865,3,1,Somerst,2006",
        "1020,Y,1,180,1,1,Crawfor,1936",
        "902,Y,1,484,2,1,Mitchel,1983",
        "742,Y,1,390,2,1,Gilbert,2005",
        "1105,Y,0,480,2,1,NAmes,1960",
        "1268,Y,1,252,1,1,CollgCr,1977",
        "1015,N,1,450,1,1,BrkSide,1925",
        "1001,N,0,871,3,1,Crawfor,1949",
        "612,Y,1,528,2,1,Somerst,2009",
        "546,Y,1,286,1,1,MeadowV,1970",
        "480,N,0,308,1,1,IDOTRR,1949",
        "1229,Y,0,284,1,1,Edwards,1956",
        "912,Y,1,833,3,1,SawyerW,1991",
        "1414,Y,1,601,2,1,NAmes,1958",
        "936,Y,1,471,2,1,SawyerW,1994",
        "1272,Y,1,0,0,1,Edwards,1900",
        "1316,Y,1,397,2,1,Gilbert,2005",
        "1617,Y,1,533,2,1,SawyerW,1993",
        "1686,Y,1,612,2,1,NWAmes,1980",
        "1126,Y,1,540,2,1,NWAmes,1977",
        "1234,Y,1,656,3,1,NridgHt,2003",
        "1098,Y,1,486,2,1,NWAmes,1968",
        "1788,Y,1,522,2,1,CollgCr,2001",
        "993,Y,1,642,2,1,Gilbert,1997",
        "1466,Y,1,610,3,1,Timber,2007",
        "925,Y,1,429,1,1,NAmes,1965",
        "1905,Y,1,788,3,1,Somerst,2006",
        "1500,Y,1,570,2,1,CollgCr,2004",
        "2069,Y,1,505,2,1,NAmes,1960",
        "747,Y,1,528,2,1,BrkSide,1926",
        "1200,Y,1,555,2,1,SawyerW,2003",
        "1113,Y,1,689,2,1,Somerst,2005",
        "1391,Y,1,868,3,1,CollgCr,2006",
        "1207,Y,1,349,1,1,ClearCr,1940",
        "1728,Y,1,574,2,1,Mitchel,1976",
        "1022,Y,1,390,2,1,Gilbert,1999",
        "1440,Y,1,0,0,1,Edwards,1977",
        "1632,Y,1,576,2,1,Mitchel,1967",
        "1344,Y,0,525,2,1,Timber,1958",
        "1188,Y,1,456,2,1,OldTown,1890",
        "1144,Y,1,796,1,1,NAmes,1959",
        "1629,Y,1,808,3,1,Timber,2002",
        "936,Y,1,474,2,1,CollgCr,2002",
        "1381,Y,1,676,2,1,NWAmes,1972",
        "864,Y,1,720,2,1,NAmes,1950",
        "965,Y,1,300,1,1,SawyerW,1965",
        "768,Y,1,396,1,1,CollgCr,1972",
        "1168,Y,1,530,2,1,Mitchel,1969",
        "980,Y,1,0,0,1,Edwards,1975",
        "979,N,0,492,2,1,Crawfor,1946",
        "561,Y,1,462,2,1,Blueste,1980",
        "1057,Y,1,576,2,1,Sawyer,1962",
        "1337,Y,1,531,2,1,CollgCr,2003",
        "696,Y,1,484,2,1,Somerst,1999",
        "858,Y,1,0,0,1,BrkSide,1958",
        "1542,Y,1,619,2,1,NWAmes,1977",
        "804,Y,1,440,2,1,NPkVill,1976",
        "1800,Y,1,702,2,1,CollgCr,2007",
        "824,Y,1,510,2,1,Timber,2002",
        "783,Y,1,393,2,1,Gilbert,2005",
        "976,Y,1,256,1,1,Crawfor,1940",
        "1098,Y,1,260,1,1,NAmes,1955",
        "600,N,1,0,0,0,OldTown,1910",
        "1095,Y,1,264,1,1,NAmes,1958",
        "720,N,0,0,0,1,NAmes,1949",
        "764,Y,1,474,2,1,NridgHt,2003",
        "918,Y,1,264,1,1,SawyerW,1979",
        "1428,Y,1,480,2,1,Somerst,2007",
        "1136,N,1,532,2,1,BrkSide,1910",
        "673,Y,1,490,2,1,Somerst,2000",
        "869,Y,0,0,0,1,BrkSide,1923",
        "1241,Y,1,569,2,1,Somerst,2006",
        "894,Y,1,400,2,1,Edwards,1954",
        "1121,Y,1,480,2,1,Sawyer,1963",
        "999,Y,1,588,2,1,NAmes,1961",
        "1276,Y,1,676,3,1,NoRidge,1998",
        "1266,Y,1,388,2,1,Blmngtn,2007",
        "1149,Y,1,779,2,1,CollgCr,2002",
        "1302,Y,1,539,2,1,Mitchel,1977",
        "1164,N,1,240,1,1,Edwards,1950",
        "1001,Y,1,255,1,1,OldTown,1910",
        "1940,Y,1,606,3,1,NridgHt,2009",
        "1118,Y,1,551,2,1,NWAmes,1976",
        "778,Y,1,614,2,1,Somerst,2006",
        "1407,Y,1,870,3,1,NoRidge,1997",
        "916,Y,1,424,2,1,OldTown,1882",
        "1020,Y,1,440,2,1,NAmes,1964",
        "750,Y,1,564,2,1,CollgCr,2005",
        "1718,Y,1,786,3,1,NridgHt,2006",
        "774,Y,1,305,1,1,BrkSide,1946",
        "1050,Y,1,368,1,1,NAmes,1961",
        "1442,Y,1,615,2,1,NWAmes,1970",
        "1077,N,1,210,1,1,IDOTRR,1922",
        "1208,Y,1,632,2,1,CollgCr,2006",
        "944,N,0,528,2,1,Edwards,1952",
        "691,Y,0,216,1,1,OldTown,1920",
        "1574,Y,1,824,3,1,Somerst,2006",
        "1680,Y,1,528,2,1,NWAmes,1976",
        "1504,Y,1,457,2,1,Blmngtn,2005",
        "985,Y,1,328,1,1,CollgCr,1977",
        "1657,Y,1,484,2,1,NAmes,1970",
        "546,Y,1,286,1,1,MeadowV,1970",
        "1710,Y,1,550,2,1,Mitchel,2004",
        "1008,Y,1,0,0,1,SWISU,1926",
        "720,Y,1,312,1,1,Sawyer,1948",
        "1664,N,1,0,0,1,Edwards,1965",
        "900,Y,1,180,1,1,Crawfor,1923",
        "1022,N,1,280,1,1,OldTown,1910",
        "1082,Y,1,240,1,1,NAmes,1948",
        "810,Y,1,528,2,1,NWAmes,2001",
        "1504,Y,1,478,2,1,CollgCr,1996",
        "1360,Y,1,565,2,1,StoneBr,1984",
        "802,Y,1,402,2,1,Gilbert,1991",
        "1506,Y,1,440,2,1,Blmngtn,2005",
        "1132,Y,1,451,2,1,Edwards,2005",
        "1220,Y,1,632,2,1,CollgCr,2006",
        "912,Y,0,160,1,1,OldTown,1930",
        "1504,Y,1,437,2,1,Blmngtn,2005",
        "2898,Y,1,665,2,1,Timber,1976",
        "882,Y,1,461,2,1,CollgCr,1972",
        "1264,Y,1,461,2,1,NAmes,1960",
        "1646,Y,1,800,3,1,Timber,2007",
        "968,Y,1,240,1,1,NAmes,1941",
        "672,Y,1,264,1,1,BrDale,1972",
        "948,Y,1,0,0,1,SWISU,1916",
        "1687,Y,1,672,2,1,SWISU,1920",
        "1352,Y,1,796,3,1,NoRidge,1993",
        "1654,Y,1,900,3,1,CollgCr,2002",
        "954,Y,1,240,1,1,Crawfor,1938",
        "845,N,1,290,1,1,Edwards,1957",
        "1620,Y,1,912,3,1,Timber,2007",
        "1055,Y,1,905,2,1,CollgCr,2001",
        "798,Y,1,0,0,1,MeadowV,1970",
        "630,Y,1,286,1,1,MeadowV,1970",
        "1803,Y,1,484,2,1,Sawyer,1957",
        "800,Y,1,484,2,1,NWAmes,1966",
        "1306,Y,1,624,2,1,NridgHt,2005",
        "1532,Y,1,514,2,1,SawyerW,1990",
        "2524,Y,1,542,2,1,NWAmes,1981",
        "1733,Y,0,452,2,1,Edwards,1955",
        "1992,Y,1,716,3,1,StoneBr,2005",
        "990,Y,1,672,2,1,CollgCr,1994",
        "1771,Y,1,336,2,1,Mitchel,1960",
        "930,Y,1,308,1,1,Edwards,1946",
        "1302,Y,1,436,2,1,Gilbert,2007",
        "1316,Y,1,440,2,1,CollgCr,2007",
        "1127,Y,1,540,2,1,NAmes,1964",
        "1526,Y,0,364,1,1,Edwards,1957",
        "1091,Y,1,586,2,1,CollgCr,2002",
        "1523,Y,1,478,2,1,NWAmes,1976",
        "1364,Y,1,484,2,1,NridgHt,2005",
        "979,Y,1,467,2,1,NoRidge,1994",
        "1130,Y,1,836,3,1,NridgHt,2008",
        "1096,Y,1,432,2,1,Crawfor,1932",
        "1338,Y,1,582,2,1,StoneBr,2001",
        "894,Y,1,1248,3,1,IDOTRR,1935",
        "1422,N,1,560,2,1,OldTown,1900",
        "1103,Y,1,440,2,1,OldTown,1925",
        "1154,Y,1,480,2,1,NAmes,1966",
        "1306,Y,1,533,2,1,ClearCr,1996",
        "799,Y,1,380,2,1,Gilbert,1993",
        "798,Y,1,442,2,1,NAmes,1964",
        "1291,Y,1,576,2,1,MeadowV,1973",
        "893,Y,1,576,2,1,NAmes,1949",
        "1048,Y,1,286,1,1,NAmes,1956",
        "829,Y,1,441,2,1,NWAmes,1968",
        "1002,Y,0,280,1,1,Edwards,1948",
        "698,Y,1,440,2,1,Edwards,1977",
        "1240,Y,1,826,3,1,CollgCr,2006",
        "960,Y,1,240,1,1,Crawfor,1940",
        "1096,Y,1,566,2,1,OldTown,1936",
        "1096,Y,1,299,1,1,NAmes,1969",
        "848,Y,1,420,2,1,CollgCr,2004",
        "990,Y,1,299,1,1,CollgCr,1994",
        "1258,Y,1,528,2,1,NWAmes,1971",
        "1040,Y,1,308,1,1,Sawyer,1963",
        "1459,Y,1,527,2,1,CollgCr,2002",
        "1251,Y,1,461,1,1,NAmes,1964",
        "691,Y,1,409,2,1,Gilbert,1995",
        "996,Y,1,564,2,1,SawyerW,1992",
        "546,Y,1,286,1,1,MeadowV,1973",
        "1082,Y,1,1043,3,1,Somerst,2005",
        "970,Y,1,380,2,1,Edwards,2004",
        "1247,Y,1,550,2,1,Somerst,2005",
        "1040,N,0,400,2,0,NAmes,1950",
        "624,Y,1,462,2,1,Somerst,1999",
        "1390,Y,1,576,2,1,SWISU,1925",
        "1200,Y,1,884,2,1,NAmes,1965",
        "936,Y,1,308,1,1,NAmes,1956",
        "1314,Y,1,440,2,1,CollgCr,2006",
        "773,Y,1,0,0,1,IDOTRR,1914",
        "1088,Y,1,461,2,1,StoneBr,1986",
        "757,Y,1,240,1,1,BrkSide,1936",
        "1601,Y,1,478,2,1,NWAmes,1978",
        "438,N,0,246,1,1,SWISU,1920",
        "950,Y,1,280,1,1,NAmes,1971",
        "1134,Y,1,254,1,1,NAmes,1960",
        "1194,Y,1,539,2,1,NAmes,1959",
        "630,Y,1,440,2,1,BrDale,1970",
        "1500,Y,1,712,2,1,NoRidge,1994",
        "1442,Y,1,719,2,1,SawyerW,1990",
        "887,Y,1,422,2,1,Gilbert,2006",
        "948,Y,1,463,2,1,Gilbert,2000",
        "1836,Y,1,862,3,1,NridgHt,2004",
        "773,Y,1,431,2,1,Gilbert,1995",
        "1098,Y,1,483,2,1,NWAmes,1976",
        "816,Y,0,308,1,1,NAmes,1957",
        "1008,Y,1,240,1,1,NAmes,1953",
        "833,Y,1,326,1,1,OldTown,1954",
        "1734,Y,1,928,3,1,NridgHt,2007",
        "779,Y,1,527,2,1,Gilbert,2002",
        "894,Y,1,450,2,1,Sawyer,1967",
        "1021,Y,1,300,1,1,Sawyer,1958",
        "1040,Y,0,286,1,1,NAmes,1959",
        "1012,Y,0,308,1,1,IDOTRR,1920",
        "1552,Y,1,782,3,1,CollgCr,2005",
        "960,Y,0,288,1,1,Edwards,1956",
        "698,Y,1,0,0,1,Edwards,1947",
        "812,Y,1,392,2,1,Gilbert,1992",
        "1005,Y,1,672,2,1,NAmes,1955",
        "1555,Y,1,660,3,1,Blmngtn,2007",
        "1530,Y,1,630,3,1,Gilbert,2004",
        "847,Y,1,434,2,1,Gilbert,2004",
        "936,Y,1,672,2,1,OldTown,1980",
        "1328,Y,1,576,2,1,SWISU,1928",
        "974,Y,1,0,0,1,Mitchel,1991",
        "1178,Y,1,205,1,1,OldTown,1880",
        "1142,Y,1,466,2,1,SawyerW,1995",
        "916,Y,1,460,2,1,Gilbert,1997",
        "986,N,1,180,1,1,BrkSide,1926",
        "1032,Y,0,288,1,1,NAmes,1950",
        "780,N,1,0,0,1,Sawyer,1875",
        "1567,Y,1,714,2,1,Mitchel,1977",
        "1167,Y,1,495,2,1,BrkSide,1920",
        "952,Y,1,840,2,1,NAmes,1951",
        "1088,Y,1,484,2,1,NWAmes,1976",
        "1466,Y,1,1052,3,1,CollgCr,2006",
        "1006,Y,1,0,0,1,Sawyer,1959",
        "672,N,0,280,1,0,Edwards,1941",
        "1042,Y,1,225,1,1,BrkSide,1928",
        "1298,Y,1,403,2,1,SawyerW,1985",
        "860,Y,1,234,1,1,Crawfor,1941",
        "572,Y,1,288,1,1,OldTown,1926",
        "832,Y,1,324,2,1,OldTown,1920",
        "932,Y,0,306,1,1,NAmes,1950",
        "1466,Y,1,528,2,1,Edwards,1959",
        "1811,Y,1,470,2,1,Crawfor,1956",
        "816,Y,1,432,1,1,IDOTRR,1930",
        "902,Y,1,492,2,1,NAmes,1965",
        "1437,Y,1,528,2,1,Veenker,1976",
        "1265,Y,1,502,2,1,NAmes,1965",
        "1314,Y,1,626,2,1,NridgHt,2007",
        "1580,Y,1,830,3,1,Somerst,2007",
        "943,Y,1,540,2,1,NWAmes,1974",
        "855,Y,1,440,2,1,NPkVill,1978",
        "1640,Y,1,924,2,1,Crawfor,1954",
        "894,Y,1,450,2,1,Sawyer,1968",
        "1258,Y,1,400,2,1,Sawyer,1969",
        "1432,Y,1,588,2,1,Veenker,1978",
        "1502,Y,1,644,2,1,NridgHt,2009",
        "1694,Y,1,776,3,1,CollgCr,2008",
        "959,Y,1,472,2,1,Gilbert,2000",
        "1236,Y,1,540,2,1,Edwards,1935",
        "1831,Y,1,807,3,1,NoRidge,1995",
        "1118,Y,1,358,1,1,Mitchel,1977",
        "1261,Y,1,433,2,1,NAmes,1958",
        "625,Y,1,625,2,1,Somerst,2006",
        "1636,Y,1,0,0,1,ClearCr,1946",
        "1170,Y,1,360,2,1,Crawfor,1932",
        "2129,Y,1,541,2,1,NoRidge,1992",
        "923,Y,1,264,1,1,Mitchel,1984",
        "818,Y,1,210,1,1,OldTown,1926",
        "820,Y,1,186,1,1,Crawfor,1921",
        "1124,Y,1,0,0,1,Edwards,1954",
        "1298,Y,1,693,2,1,Timber,1990",
        "1652,Y,1,482,2,1,Crawfor,2008",
        "2411,Y,1,813,3,1,NoRidge,1996",
        "1130,Y,1,720,2,1,OldTown,1920",
        "1572,Y,1,995,3,1,Timber,1963",
        "949,Y,1,392,1,1,Edwards,1924",
        "1014,Y,1,420,2,1,OldTown,1900",
        "1624,Y,1,757,3,1,NoRidge,1994",
        "831,Y,1,493,2,1,CollgCr,2002",
        "1028,Y,1,442,2,1,Gilbert,1999",
        "1622,Y,1,1356,4,1,Mitchel,1961",
        "764,Y,1,492,2,1,Somerst,1999",
        "842,Y,1,250,1,1,OldTown,1925",
        "1224,Y,1,402,2,1,Mitchel,1999",
        "663,Y,1,299,1,1,Sawyer,1969",
        "728,Y,1,400,2,1,Gilbert,2005",
        "879,Y,1,660,3,1,Gilbert,2006",
        "815,Y,1,225,1,1,OldTown,1916",
        "1212,Y,1,573,2,1,CollgCr,2001",
        "1382,Y,1,459,2,1,Sawyer,1963",
        "864,Y,1,280,1,1,NAmes,1970",
        "866,Y,1,546,2,1,CollgCr,1998",
        "884,Y,1,216,1,1,BrkSide,1925",
        "1630,Y,1,451,2,1,CollgCr,2000",
        "1074,Y,1,495,2,1,NWAmes,1975",
        "2196,Y,1,701,3,1,SawyerW,1990",
        "1056,Y,1,384,1,1,SawyerW,1966",
        "1700,Y,1,544,2,1,CollgCr,2003",
        "1283,Y,1,506,2,1,NAmes,1962",
        "1660,Y,1,500,2,1,Somerst,2006",
        "1055,Y,1,462,2,1,SawyerW,1992",
        "1080,Y,1,492,2,1,Gilbert,1988",
        "672,Y,1,234,1,1,Edwards,1941",
        "960,Y,1,364,1,1,Sawyer,1965",
        "999,Y,1,300,1,1,NAmes,1962",
        "894,Y,1,384,1,1,Sawyer,1966",
        "1318,Y,1,539,2,1,Sawyer,1978",
        "1314,Y,1,552,2,1,Somerst,2009",
        "672,N,1,0,0,1,BrkSide,1947",
        "672,Y,1,0,0,1,BrDale,1971",
        "912,Y,1,288,1,1,NAmes,1964",
        "1211,Y,1,322,1,1,Sawyer,1968",
        "1168,Y,1,315,1,1,NAmes,1949",
        "2136,N,0,528,2,1,NAmes,1951",
        "788,Y,1,388,2,1,Gilbert,2004",
        "1138,Y,1,264,1,1,NAmes,1958",
        "894,Y,1,668,3,1,Somerst,2007",
        "912,Y,1,576,2,1,NAmes,1965",
        "1702,Y,1,1052,3,1,NridgHt,2008",
        "1507,Y,1,404,1,1,NAmes,1960",
        "1361,Y,1,600,2,1,Sawyer,1977",
        "1190,Y,1,540,2,1,NAmes,1962",
        "1224,Y,1,462,2,1,NAmes,1962",
        "1188,Y,1,531,2,1,NAmes,1959",
        "1024,N,0,0,0,1,SWISU,1911",
        "892,Y,0,180,1,1,Crawfor,1914",
        "764,Y,1,474,2,1,NridgHt,2003",
        "847,Y,1,434,2,1,CollgCr,2004",
        "1141,Y,1,484,2,1,SawyerW,2005",
        "1484,Y,1,472,2,1,Timber,2006",
        "884,Y,1,543,2,1,CollgCr,2003",
        "1689,Y,1,954,3,1,Somerst,2007",
        "1173,Y,1,528,2,1,NWAmes,1974",
        "2076,Y,1,850,3,1,NridgHt,2006",
        "792,Y,1,400,2,1,Crawfor,1929",
        "1140,Y,1,477,2,1,NWAmes,1984",
        "756,Y,1,615,2,1,Somerst,2005",
        "1034,Y,1,888,3,1,Mitchel,1976",
        "1134,N,1,396,2,1,OldTown,1917",
        "988,Y,1,276,1,1,NAmes,1950",
        "2110,Y,1,522,2,1,NAmes,1968",
        "1405,Y,1,478,2,1,NridgHt,2003",
        "874,Y,1,288,1,1,Mitchel,1968",
        "1516,Y,1,518,2,1,Veenker,1974",
        "760,Y,1,397,2,1,Edwards,2003",
        "959,Y,0,560,1,1,BrkSide,1931",
        "1987,Y,1,691,2,1,NoRidge,1994",
        "864,Y,1,0,0,1,Edwards,1922",
        "1166,Y,1,400,2,1,Gilbert,2005",
        "1054,Y,1,460,2,1,NAmes,1969",
        "892,Y,1,502,2,1,Gilbert,1999",
        "1050,Y,1,338,1,1,NAmes,1956",
        "1104,Y,0,304,1,1,ClearCr,1957",
        "1060,Y,0,520,2,1,BrkSide,1919",
        "1337,Y,1,511,2,1,NAmes,1998",
        "713,Y,1,506,2,1,Somerst,1999",
        "964,N,0,308,1,1,OldTown,1910",
        "2018,Y,1,746,3,1,Timber,2008",
        "1968,Y,1,1014,3,1,Crawfor,1935",
        "874,Y,0,315,1,1,NAmes,1958",
        "1332,Y,1,586,2,1,ClearCr,1979",
        "1489,Y,1,462,2,1,NWAmes,1968",
        "935,Y,1,288,1,1,Sawyer,1965",
        "1357,Y,1,312,1,1,Edwards,1959",
        "661,Y,1,552,2,1,Crawfor,1910",
        "928,Y,0,400,2,1,NAmes,1948",
        "735,Y,1,497,2,1,NWAmes,1972",
        "1724,Y,1,480,2,1,NWAmes,1967",
        "1128,Y,1,577,2,1,CollgCr,2002",
        "698,Y,0,528,2,1,IDOTRR,1920",
        "1573,Y,1,544,2,1,CollgCr,2002",
        "1339,Y,1,484,2,1,Timber,1990",
        "1040,Y,1,484,2,1,CollgCr,1977",
        "912,Y,1,0,0,1,Mitchel,1971",
        "1699,Y,1,336,1,1,SWISU,1919",
        "825,Y,0,280,1,1,BrkSide,1939",
        "1328,Y,1,528,2,1,NWAmes,1963",
        "1582,Y,1,390,2,1,ClearCr,1964",
        "1659,Y,1,499,2,1,StoneBr,2000",
        "1120,Y,1,753,3,1,NridgHt,2006",
        "1152,Y,1,484,2,1,NAmes,1964",
        "630,Y,1,264,1,1,BrDale,1972",
        "1378,N,1,432,1,1,OldTown,1892",
        "832,Y,1,528,2,1,NWAmes,1976",
        "864,Y,1,572,2,1,Edwards,1955",
        "1052,Y,1,288,1,1,NAmes,1968",
        "1128,Y,1,525,2,1,NAmes,1963",
        "1072,Y,1,525,2,1,Edwards,2005",
        "4692,Y,1,1418,2,1,Edwards,2008",
        "1246,Y,1,305,1,1,NAmes,1959",
        "1005,Y,1,490,2,1,Gilbert,1999",
        "753,Y,1,213,1,1,Crawfor,1942",
        "1203,Y,1,844,3,1,NoRidge,1994",
        "1616,Y,1,834,3,1,Somerst,2005",
        "976,Y,1,380,2,1,Edwards,2004",
        "1652,Y,1,840,2,1,NridgHt,2006",
        "1368,Y,1,474,2,1,NridgHt,2005",
        "990,Y,1,480,2,1,CollgCr,1994",
        "1122,Y,1,528,2,1,OldTown,1948",
        "1294,Y,1,496,2,1,SawyerW,1991",
        "1902,Y,1,567,2,1,Crawfor,1959",
        "1274,Y,1,508,2,1,CollgCr,2005",
        "1453,Y,1,750,2,1,NoRidge,1990",
        "1422,Y,1,779,3,1,NoRidge,1999",
        "948,Y,1,280,1,1,Edwards,1954",
        "1092,Y,1,576,2,1,NAmes,1969",
        "1630,Y,1,860,3,1,CollgCr,2008",
        "1352,Y,1,466,2,1,Somerst,2006",
        "1787,Y,1,748,3,1,CollgCr,2001",
        "948,Y,1,248,1,1,Edwards,1954",
        "1478,Y,1,442,2,1,NAmes,1957",
        "720,N,1,287,1,0,BrkSide,1949",
        "1061,Y,1,564,2,1,NoRidge,1992",
        "708,Y,1,0,0,1,BrkSide,1940",
        "1795,Y,1,895,3,1,Somerst,2006",
        "796,N,1,0,0,1,IDOTRR,1922",
        "774,Y,1,0,0,1,Edwards,1931",
        "816,Y,1,264,1,1,Mitchel,1982",
        "1584,Y,1,520,2,1,OldTown,1920",
        "955,Y,1,462,2,1,Gilbert,1998",
        "1588,Y,1,825,3,1,Somerst,2006",
        "954,Y,1,576,2,1,CollgCr,1976",
        "816,Y,1,288,1,1,Edwards,1938",
        "803,Y,1,297,1,1,IDOTRR,1938",
        "765,Y,1,440,2,1,BrDale,1970",
        "1334,Y,1,630,2,1,NWAmes,1977",
        "1656,Y,1,506,2,1,NWAmes,1973",
        "693,N,0,0,0,0,OldTown,1941",
        "920,Y,1,492,2,1,CollgCr,2002",
        "864,Y,1,288,1,1,CollgCr,1972",
        "872,Y,1,480,4,1,NAmes,1971",
        "1114,Y,1,576,2,1,SawyerW,2003",
        "1284,Y,1,647,2,1,CollgCr,2002",
        "1172,Y,1,342,2,1,Crawfor,1928",
        "728,Y,1,440,2,1,CollgCr,2006",
        "960,Y,0,308,1,1,OldTown,1920",
        "2156,Y,1,508,2,1,NWAmes,1968",
        "1776,Y,1,712,3,1,Timber,2006",
        "1494,Y,1,514,2,1,SawyerW,1998",
        "938,N,1,0,0,1,OldTown,1872",
        "1338,Y,1,968,4,1,NAmes,1969",
        "858,Y,1,490,2,1,NAmes,1962",
        "786,Y,1,624,2,1,BrkSide,1937",
        "2053,Y,1,666,3,1,NoRidge,1995",
        "992,Y,1,839,3,1,CollgCr,2000",
        "1222,Y,1,487,2,1,NWAmes,1968",
        "892,Y,1,264,1,1,NAmes,1966",
        "1078,Y,1,500,2,1,NAmes,1971",
        "769,Y,1,440,2,1,Somerst,2000",
        "1980,Y,1,770,3,1,NridgHt,2004",
        "990,Y,1,621,2,1,SWISU,1921",
        "1530,Y,1,430,2,1,StoneBr,2005",
        "1281,Y,1,368,1,1,NAmes,1920",
        "616,Y,1,432,2,1,Gilbert,2006",
        "520,Y,1,480,2,1,Somerst,2005",
        "814,Y,1,663,2,1,Somerst,2000",
        "882,Y,1,588,2,1,CollgCr,1999",
        "925,Y,1,336,1,1,MeadowV,1977",
        "848,Y,1,420,2,1,CollgCr,2003",
        "1668,Y,1,502,2,1,CollgCr,2003",
        "840,Y,1,338,1,1,OldTown,1920",
        "1661,Y,1,377,1,1,NAmes,1955",
        "1108,Y,1,583,2,1,CollgCr,1998",
        "2633,Y,1,804,3,1,NoRidge,2001",
        "1026,Y,1,936,3,1,Somerst,2005",
        "1571,Y,1,722,3,1,Timber,2007",
        "790,N,1,160,1,1,SWISU,1930",
        "984,Y,1,660,2,1,Edwards,1941",
        "483,Y,1,264,1,1,BrDale,1973",
        "754,Y,1,400,2,1,Timber,2006",
        "864,N,0,200,1,1,Edwards,1914",
        "2117,Y,1,550,2,1,NAmes,1970",
        "998,Y,1,576,2,1,OldTown,1920",
        "1416,Y,1,576,2,1,Sawyer,1918",
        "698,Y,1,280,1,1,Edwards,1939",
        "796,Y,1,240,1,1,IDOTRR,1922",
        "1392,Y,1,564,2,1,NWAmes,1978",
        "1664,N,1,216,1,1,OldTown,1916",
        "1746,Y,1,758,3,1,Gilbert,2006",
        "869,Y,1,440,2,1,BrkSide,1941",
        "1525,Y,1,541,2,1,CollgCr,2000",
        "1584,Y,1,792,3,1,NAmes,1967",
        "900,Y,1,288,1,1,NAmes,1967",
        "1221,N,1,672,2,1,OldTown,1905",
        "1500,Y,1,648,3,1,Blmngtn,2006",
        "1133,Y,1,642,3,1,Timber,2005",
        "1687,Y,1,572,2,1,Timber,1948",
        "939,Y,1,180,1,1,BrkSide,1920",
        "1136,Y,0,240,1,1,NAmes,1950",
        "1160,Y,1,216,1,1,SWISU,1925",
        "950,Y,1,208,1,1,BrkSide,1929",
        "864,Y,1,398,2,1,Gilbert,2004",
        "1294,Y,1,662,2,1,CollgCr,2006",
        "1464,Y,1,754,3,1,Somerst,2007",
        "694,Y,1,936,3,1,OldTown,1915",
        "1646,Y,1,482,2,1,Crawfor,2004",
        "768,Y,1,396,2,1,CollgCr,1972",
        "833,Y,1,0,0,1,Mitchel,1985",
        "741,Y,1,528,2,1,OldTown,1910",
        "1236,Y,1,542,2,1,NWAmes,1986",
        "944,Y,1,622,2,1,CollgCr,2001",
        "1112,Y,1,271,1,1,NAmes,1950",
        "1040,N,0,420,2,0,NAmes,1949",
        "1844,Y,1,620,2,1,SawyerW,2005",
        "1053,Y,1,370,2,1,BrkSide,1923",
        "1569,Y,1,660,3,1,Blmngtn,2007",
        "1246,Y,1,560,2,1,OldTown,1885",
        "1310,Y,1,1069,3,1,NoRidge,1998",
        "1144,Y,1,336,1,1,NAmes,1963",
        "1844,Y,1,540,2,1,Crawfor,1969",
        "708,Y,1,776,2,1,NWAmes,1968",
        "1069,Y,1,440,2,1,NPkVill,1977",
        "848,Y,1,420,2,1,CollgCr,2003",
        "1575,Y,1,432,2,1,Edwards,1966",
        "1344,Y,1,484,1,1,NAmes,1958",
        "1252,Y,1,528,2,1,NAmes,1959",
        "1223,Y,1,525,2,1,NoRidge,1994",
        "1048,Y,0,288,1,1,NAmes,1945",
        "804,Y,1,240,1,1,OldTown,1940",
        "1440,Y,1,467,2,1,NWAmes,1981",
        "734,Y,1,372,2,1,Gilbert,2005",
        "958,Y,1,440,2,1,NPkVill,1976",
        "968,Y,1,216,1,1,OldTown,1927",
        "962,Y,1,451,2,1,Gilbert,2000",
        "1126,Y,1,484,2,1,Mitchel,1977",
        "1537,Y,1,462,2,1,NAmes,1962",
        "864,Y,1,528,2,1,NAmes,1971",
        "1932,Y,1,774,3,1,NridgHt,2008",
        "1236,Y,0,923,2,1,OldTown,1957",
        "1040,Y,1,550,2,1,NWAmes,1979",
        "1423,Y,1,672,2,1,Crawfor,1922",
        "848,Y,1,420,2,1,CollgCr,2004",
        "1026,Y,1,812,3,1,Somerst,2008",
        "952,N,0,192,1,0,BrkSide,1916",
        "1422,Y,1,626,2,1,CollgCr,2004",
        "913,Y,1,240,1,1,Sawyer,1966",
        "1188,Y,1,312,1,1,Mitchel,1962",
        "1220,Y,1,556,2,1,CollgCr,1995",
        "796,N,1,384,1,1,Edwards,1910",
        "630,Y,1,0,0,1,MeadowV,1970",
        "896,Y,1,0,0,1,NAmes,1974",
        "1578,Y,1,840,3,1,Somerst,2008",
        "1072,Y,1,525,2,1,Edwards,2005",
        "1140,Y,1,0,0,1,Mitchel,2006",
        "1221,Y,1,400,2,1,Somerst,2004",
        "953,Y,1,460,2,1,Gilbert,1999",
        "2073,Y,1,500,2,1,NWAmes,1978",
        "1188,Y,1,252,1,1,Crawfor,1941",
        "1078,Y,0,240,1,1,NAmes,1950",
        "1256,Y,1,276,1,1,Edwards,1965"
    );

    static final String TARGET_FIELD = "CentralAir";

    private String jobId;
    private String sourceIndex;
    private String destIndex;
    private long randomizeSeed;

    @Before
    public void setupLogging() {
        client().admin()
            .cluster()
            .prepareUpdateSettings()
            .setTransientSettings(
                Settings.builder()
                    .put("logger.org.elasticsearch.xpack.ml.process", "DEBUG")
                    .put("logger.org.elasticsearch.xpack.ml.dataframe", "DEBUG")
            )
            .get();
    }

    @Before
    public void setUpTests() {
        randomizeSeed = randomLong();
    }

    @After
    public void cleanup() {
        client().admin()
            .cluster()
            .prepareUpdateSettings()
            .setTransientSettings(
                Settings.builder()
                    .putNull("logger.org.elasticsearch.xpack.ml.process")
                    .putNull("logger.org.elasticsearch.xpack.ml.dataframe")
            )
            .get();
        cleanUp();
    }

    public void testFeatureImportanceValues() throws Exception {
        initialize("classification_house_pricing_test_feature_importance_values");
        indexData(sourceIndex);
        DataFrameAnalyticsConfig config = buildAnalytics(
            jobId,
            sourceIndex,
            destIndex,
            null,
            new Classification(
                TARGET_FIELD,
                BoostedTreeParams.builder().setNumTopFeatureImportanceValues(5).build(),
                null,
                null,
                null,
                35.0,
                randomizeSeed,
                null,
                null
            )
        );

        putAnalytics(config);

        assertIsStopped(jobId);
        assertProgressIsZero(jobId);

        startAnalytics(jobId);
        waitUntilAnalyticsIsStopped(jobId);

        client().admin().indices().refresh(new RefreshRequest(destIndex));
        SearchResponse sourceData = client().prepareSearch(sourceIndex).setTrackTotalHits(true).setSize(1000).get();
        for (SearchHit hit : sourceData.getHits()) {
            Map<String, Object> destDoc = getDestDoc(config, hit);
            Map<String, Object> resultsObject = getFieldValue(destDoc, "ml");
            @SuppressWarnings("unchecked")
            List<Map<String, Object>> importanceArray = (List<Map<String, Object>>) resultsObject.get("feature_importance");
            assertThat(importanceArray, hasSize(greaterThan(0)));
        }

    }

    static void indexData(String sourceIndex) {
        BulkRequestBuilder bulkRequestBuilder = client().prepareBulk().setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        for (String row : DATA) {
            String[] values = row.split(",");
            List<Object> source = List.of(
                "1stFlrSF",
                Integer.valueOf(values[0]),
                "CentralAir",
                values[1],
                "Electrical",
                values[2],
                "GarageArea",
                Integer.valueOf(values[3]),
                "GarageCars",
                Integer.valueOf(values[4]),
                "Heating",
                values[5],
                "Neighborhood",
                values[6],
                "YearBuilt",
                Integer.valueOf(values[7])
            );
            IndexRequest indexRequest = new IndexRequest(sourceIndex).source(source.toArray()).opType(DocWriteRequest.OpType.CREATE);
            bulkRequestBuilder.add(indexRequest);
        }
        BulkResponse bulkResponse = bulkRequestBuilder.get();
        if (bulkResponse.hasFailures()) {
            fail("Failed to index data: " + bulkResponse.buildFailureMessage());
        }
    }

    private void initialize(String jobId) {
        this.jobId = jobId;
        this.sourceIndex = jobId + "_source_index";
        this.destIndex = sourceIndex + "_results";
        createIndex(sourceIndex);
    }

    static void createIndex(String index) {
        String mapping = """
            {
              "properties": {
                   "1stFlrSF": {
                     "type": "integer"
                   },
                   "CentralAir": {
                     "type": "keyword"
                   },
                   "Electrical": {
                     "type": "keyword"
                   },
                   "GarageArea": {
                     "type": "integer"
                   },
                   "GarageCars": {
                     "type": "integer"
                   },
                   "Heating": {
                     "type": "keyword"
                   },
                   "Neighborhood": {
                     "type": "keyword"
                   },
                   "YearBuilt": {
                     "type": "integer"
                   }
             }
            }""";

        client().admin().indices().prepareCreate(index).setMapping(mapping).get();
    }

    @Override
    boolean supportsInference() {
        return true;
    }

    private static Map<String, Object> getDestDoc(DataFrameAnalyticsConfig config, SearchHit hit) {
        GetResponse destDocGetResponse = client().prepareGet().setIndex(config.getDest().getIndex()).setId(hit.getId()).get();
        assertThat(destDocGetResponse.isExists(), is(true));
        Map<String, Object> sourceDoc = hit.getSourceAsMap();
        Map<String, Object> destDoc = destDocGetResponse.getSource();
        for (String field : sourceDoc.keySet()) {
            assertThat(destDoc, hasKey(field));
            assertThat(destDoc.get(field), equalTo(sourceDoc.get(field)));
        }
        return destDoc;
    }
}
