/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
package org.elasticsearch.snapshots;

import joptsimple.OptionSet;

import com.amazonaws.services.s3.internal.Constants;

import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.cli.MockTerminal;
import org.elasticsearch.common.settings.MockSecureSettings;
import org.elasticsearch.common.settings.SecureSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.plugins.Plugin;
import org.elasticsearch.repositories.s3.S3RepositoryPlugin;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import static org.hamcrest.Matchers.equalTo;

public class S3CleanupTests extends AbstractCleanupTests {

    @Override
    protected Collection<Class<? extends Plugin>> getPlugins() {
        return pluginList(S3RepositoryPlugin.class);
    }

    @Override
    protected SecureSettings credentials() {
        MockSecureSettings secureSettings = new MockSecureSettings();
        secureSettings.setString("s3.client.default.access_key", getAccessKey());
        secureSettings.setString("s3.client.default.secret_key", getSecretKey());
        return secureSettings;
    }

    @Override
    protected void createRepository(String repoName) {
        Settings.Builder settings = Settings.builder()
            .put("bucket", getBucket())
            .put("base_path", getBasePath())
            .put("endpoint", getEndpoint());

        AcknowledgedResponse putRepositoryResponse = client().admin()
            .cluster()
            .preparePutRepository(repoName)
            .setType("s3")
            .setSettings(settings)
            .get();
        assertThat(putRepositoryResponse.isAcknowledged(), equalTo(true));
    }

    private String getEndpoint() {
        return System.getProperty("test.s3.endpoint", Constants.S3_HOSTNAME);
    }

    private String getRegion() {
        return "";
    }

    private String getBucket() {
        return System.getProperty("test.s3.bucket");
    }

    private String getBasePath() {
        return System.getProperty("test.s3.base");
    }

    private String getAccessKey() {
        return System.getProperty("test.s3.account");
    }

    private String getSecretKey() {
        return System.getProperty("test.s3.key");
    }

    @Override
    protected ThrowingRunnable commandRunnable(MockTerminal terminal, Map<String, String> nonDefaultArguments) {
        final CleanupS3RepositoryCommand command = new CleanupS3RepositoryCommand();
        final OptionSet options = command.getParser()
            .parse(
                "--safety_gap_millis",
                nonDefaultArguments.getOrDefault("safety_gap_millis", "0"),
                "--parallelism",
                nonDefaultArguments.getOrDefault("parallelism", "10"),
                "--endpoint",
                nonDefaultArguments.getOrDefault("endpoint", getEndpoint()),
                "--region",
                nonDefaultArguments.getOrDefault("region", getRegion()),
                "--bucket",
                nonDefaultArguments.getOrDefault("bucket", getBucket()),
                "--base_path",
                nonDefaultArguments.getOrDefault("base_path", getBasePath()),
                "--access_key",
                nonDefaultArguments.getOrDefault("access_key", getAccessKey()),
                "--secret_key",
                nonDefaultArguments.getOrDefault("secret_key", getSecretKey())
            );
        return () -> command.execute(terminal, options);
    }

    public void testNoRegionNoEndpoint() {
        Map<String, String> args = new HashMap<>();
        args.put("region", "");
        args.put("endpoint", "");

        expectThrows(() -> executeCommand(false, args), "region or endpoint option is required for cleaning up S3 repository");
    }

    public void testRegionAndEndpointSpecified() {
        Map<String, String> args = new HashMap<>();
        args.put("region", "test_region");
        args.put("endpoint", "test_endpoint");

        expectThrows(() -> executeCommand(false, args), "you must not specify both region and endpoint");
    }

    public void testNoAccessKey() {
        expectThrows(
            () -> executeCommand(false, Collections.singletonMap("access_key", "")),
            "access_key option is required for cleaning up S3 repository"
        );
    }

    public void testNoSecretKey() {
        expectThrows(
            () -> executeCommand(false, Collections.singletonMap("secret_key", "")),
            "secret_key option is required for cleaning up S3 repository"
        );
    }
}
