/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

package org.elasticsearch.xpack.vectors.query;

import org.apache.lucene.index.BinaryDocValues;
import org.apache.lucene.index.DocValues;
import org.apache.lucene.index.LeafReader;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.BytesRef;
import org.elasticsearch.Version;
import org.elasticsearch.index.fielddata.LeafFieldData;
import org.elasticsearch.index.fielddata.ScriptDocValues;
import org.elasticsearch.index.fielddata.SortedBinaryDocValues;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;

final class VectorDVLeafFieldData implements LeafFieldData {

    private final LeafReader reader;
    private final String field;
    private final boolean isDense;
    private final Version indexVersion;
    private final int dims;

    VectorDVLeafFieldData(LeafReader reader, String field, boolean isDense, Version indexVersion, int dims) {
        this.reader = reader;
        this.field = field;
        this.isDense = isDense;
        this.indexVersion = indexVersion;
        this.dims = dims;
    }

    @Override
    public long ramBytesUsed() {
        return 0; // not exposed by Lucene
    }

    @Override
    public Collection<Accountable> getChildResources() {
        return Collections.emptyList();
    }

    @Override
    public SortedBinaryDocValues getBytesValues() {
        throw new UnsupportedOperationException("String representation of doc values for vector fields is not supported");
    }

    @Override
    public ScriptDocValues<BytesRef> getScriptValues() {
        try {
            final BinaryDocValues values = DocValues.getBinary(reader, field);
            if (isDense) {
                return new VectorScriptDocValues.DenseVectorScriptDocValues(values, indexVersion, dims);
            } else {
                return new VectorScriptDocValues.SparseVectorScriptDocValues(values, indexVersion);
            }
        } catch (IOException e) {
            throw new IllegalStateException("Cannot load doc values for vector field!", e);
        }
    }

    @Override
    public void close() {
        // no-op
    }
}
