/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

package org.elasticsearch.xpack.cluster.routing.allocation;

import org.elasticsearch.action.admin.cluster.settings.ClusterUpdateSettingsRequest;
import org.elasticsearch.action.admin.indices.settings.put.UpdateSettingsRequest;
import org.elasticsearch.action.admin.indices.shrink.ResizeType;
import org.elasticsearch.action.admin.indices.template.put.PutComposableIndexTemplateAction;
import org.elasticsearch.cluster.health.ClusterHealthStatus;
import org.elasticsearch.cluster.metadata.ComposableIndexTemplate;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Template;
import org.elasticsearch.cluster.routing.allocation.DataTier;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.Map;
import org.elasticsearch.index.mapper.MapperService;
import org.elasticsearch.plugins.Plugin;
import org.elasticsearch.test.ESIntegTestCase;
import org.elasticsearch.xpack.core.DataTiersFeatureSetUsage;
import org.elasticsearch.xpack.core.LocalStateCompositeXPackPlugin;
import org.elasticsearch.xpack.core.XPackFeatureSet;
import org.elasticsearch.xpack.core.action.XPackUsageRequestBuilder;
import org.elasticsearch.xpack.core.action.XPackUsageResponse;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.stream.Collectors;

import static org.elasticsearch.test.hamcrest.ElasticsearchAssertions.assertAcked;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;

@ESIntegTestCase.ClusterScope(scope = ESIntegTestCase.Scope.TEST, numDataNodes = 0, numClientNodes = 0, transportClientRatio = 0)
public class DataTierAllocationDeciderIT extends ESIntegTestCase {
    private static final String index = "myindex";

    @Override
    protected Collection<Class<? extends Plugin>> nodePlugins() {
        return Collections.singleton(LocalStateCompositeXPackPlugin.class);
    }

    public void testDefaultIndexAllocateToContent() {
        startWarmOnlyNode();
        startColdOnlyNode();
        ensureGreen();

        client().admin().indices().prepareCreate(index).setWaitForActiveShards(0).get();

        Settings idxSettings = client().admin().indices().prepareGetIndex().addIndices(index).get().getSettings().get(index);
        assertThat(DataTier.TIER_PREFERENCE_SETTING.get(idxSettings), equalTo(DataTier.DATA_CONTENT));

        // index should be red
        assertThat(
            client().admin().cluster().prepareHealth(index).get().getIndices().get(index).getStatus(),
            equalTo(ClusterHealthStatus.RED)
        );

        if (randomBoolean()) {
            logger.info("--> starting content node");
            startContentOnlyNode();
        } else {
            logger.info("--> starting data node");
            startDataNode();
        }

        logger.info("--> waiting for {} to be yellow", index);
        ensureYellow(index);
    }

    public void testOverrideDefaultAllocation() {
        startWarmOnlyNode();
        startColdOnlyNode();
        ensureGreen();

        client().admin()
            .indices()
            .prepareCreate(index)
            .setWaitForActiveShards(0)
            .setSettings(Settings.builder().put(DataTier.TIER_PREFERENCE, DataTier.DATA_WARM))
            .get();

        Settings idxSettings = client().admin().indices().prepareGetIndex().addIndices(index).get().getSettings().get(index);
        assertThat(idxSettings.get(DataTier.TIER_PREFERENCE), equalTo(DataTier.DATA_WARM));

        // index should be yellow
        logger.info("--> waiting for {} to be yellow", index);
        ensureYellow(index);
    }

    public void testRequestSettingOverridesAllocation() {
        startWarmOnlyNode();
        startColdOnlyNode();
        ensureGreen();
        enforceDefaultTierPreference(false);

        client().admin()
            .indices()
            .prepareCreate(index)
            .setWaitForActiveShards(0)
            .setSettings(Settings.builder().putNull(DataTier.TIER_PREFERENCE))
            .get();

        Settings idxSettings = client().admin().indices().prepareGetIndex().addIndices(index).get().getSettings().get(index);
        assertThat(DataTier.TIER_PREFERENCE_SETTING.get(idxSettings), equalTo(""));
        // Even the key shouldn't exist if it has been nulled out
        assertFalse(idxSettings.keySet().toString(), idxSettings.keySet().contains(DataTier.TIER_PREFERENCE));

        // index should be yellow
        logger.info("--> waiting for {} to be yellow", index);
        ensureYellow(index);

        client().admin().indices().prepareDelete(index).get();

        // Now test it overriding the "require" setting, in which case the preference should be skipped
        client().admin()
            .indices()
            .prepareCreate(index)
            .setWaitForActiveShards(0)
            .setSettings(Settings.builder().put(DataTierAllocationDecider.INDEX_ROUTING_REQUIRE, DataTier.DATA_COLD))
            .get();

        idxSettings = client().admin().indices().prepareGetIndex().addIndices(index).get().getSettings().get(index);
        assertThat(DataTier.TIER_PREFERENCE_SETTING.get(idxSettings), equalTo(""));
        // The key should not be put in place since it was overridden
        assertFalse(idxSettings.keySet().contains(DataTierAllocationDecider.TIER_PREFERENCE));
        assertThat(DataTierAllocationDecider.INDEX_ROUTING_REQUIRE_SETTING.get(idxSettings), equalTo(DataTier.DATA_COLD));

        // index should be yellow
        logger.info("--> waiting for {} to be yellow", index);
        ensureYellow(index);
    }

    public void testRequestSettingOverriddenIfEnforced() {
        startContentOnlyNode();
        ensureGreen();
        enforceDefaultTierPreference(true);

        client().admin()
            .indices()
            .prepareCreate(index)
            .setWaitForActiveShards(0)
            .setSettings(Settings.builder().putNull(DataTier.TIER_PREFERENCE)) // will be overridden to data_content
            .get();

        Settings idxSettings = client().admin().indices().prepareGetIndex().addIndices(index).get().getSettings().get(index);
        assertThat(DataTier.TIER_PREFERENCE_SETTING.get(idxSettings), equalTo("data_content"));

        // index should be yellow
        logger.info("--> waiting for {} to be yellow", index);
        ensureYellow(index);
    }

    /**
     * When a new index is created from source metadata (as during a shrink), the data tier
     * default setting should *not* be applied. This test checks that behavior.
     */
    public void testShrinkStaysOnTier() {
        startWarmOnlyNode();
        startHotOnlyNode();

        client().admin()
            .indices()
            .prepareCreate(index)
            .setWaitForActiveShards(0)
            .setSettings(
                Settings.builder()
                    .put(IndexMetadata.SETTING_NUMBER_OF_SHARDS, 2)
                    .put(IndexMetadata.SETTING_NUMBER_OF_REPLICAS, 0)
                    .put(DataTier.TIER_PREFERENCE, "data_warm")
            )
            .get();

        client().admin().indices().prepareAddBlock(IndexMetadata.APIBlock.READ_ONLY, index).get();
        client().admin()
            .indices()
            .prepareResizeIndex(index, index + "-shrunk")
            .setResizeType(ResizeType.SHRINK)
            .setSettings(
                Settings.builder().put(IndexMetadata.SETTING_NUMBER_OF_SHARDS, 1).put(IndexMetadata.SETTING_NUMBER_OF_REPLICAS, 0).build()
            )
            .get();

        ensureGreen(index + "-shrunk");

        Settings idxSettings = client().admin()
            .indices()
            .prepareGetIndex()
            .addIndices(index + "-shrunk")
            .get()
            .getSettings()
            .get(index + "-shrunk");
        // It should inherit the setting of its originator
        assertThat(DataTier.TIER_PREFERENCE_SETTING.get(idxSettings), equalTo(DataTier.DATA_WARM));

        // Required or else the test cleanup fails because it can't delete the indices
        client().admin()
            .indices()
            .prepareUpdateSettings(index, index + "-shrunk")
            .setSettings(Settings.builder().put("index.blocks.read_only", false))
            .get();
    }

    public void testTemplateOverridesDefaults() {
        startWarmOnlyNode();
        enforceDefaultTierPreference(false);

        Template t = new Template(
            Settings.builder().put(DataTierAllocationDecider.INDEX_ROUTING_REQUIRE, DataTier.DATA_WARM).build(),
            null,
            null
        );
        ComposableIndexTemplate ct = new ComposableIndexTemplate.Builder().indexPatterns(Collections.singletonList(index))
            .template(t)
            .build();
        client().execute(
            PutComposableIndexTemplateAction.INSTANCE,
            new PutComposableIndexTemplateAction.Request("template").indexTemplate(ct)
        ).actionGet();

        client().admin().indices().prepareCreate(index).setWaitForActiveShards(0).get();

        Settings idxSettings = client().admin().indices().prepareGetIndex().addIndices(index).get().getSettings().get(index);
        assertThat(idxSettings.keySet().contains(DataTier.TIER_PREFERENCE), equalTo(false));

        // index should be yellow
        ensureYellow(index);

        client().admin().indices().prepareDelete(index).get();

        t = new Template(Settings.builder().putNull(DataTier.TIER_PREFERENCE).build(), null, null);
        ct = new ComposableIndexTemplate.Builder().indexPatterns(Collections.singletonList(index)).template(t).build();
        client().execute(
            PutComposableIndexTemplateAction.INSTANCE,
            new PutComposableIndexTemplateAction.Request("template").indexTemplate(ct)
        ).actionGet();

        client().admin().indices().prepareCreate(index).setWaitForActiveShards(0).get();

        idxSettings = client().admin().indices().prepareGetIndex().addIndices(index).get().getSettings().get(index);
        assertThat(idxSettings.keySet().contains(DataTier.TIER_PREFERENCE), equalTo(false));

        ensureYellow(index);
    }

    public void testTemplateOverriddenIfEnforced() {
        startContentOnlyNode();
        enforceDefaultTierPreference(true);

        Template t = new Template(Settings.builder().putNull(DataTier.TIER_PREFERENCE).build(), null, null);
        ComposableIndexTemplate ct = new ComposableIndexTemplate.Builder().indexPatterns(Collections.singletonList(index))
            .template(t)
            .build();
        client().execute(
            PutComposableIndexTemplateAction.INSTANCE,
            new PutComposableIndexTemplateAction.Request("template").indexTemplate(ct)
        ).actionGet();

        client().admin().indices().prepareCreate(index).setWaitForActiveShards(0).get();

        Settings idxSettings = client().admin().indices().prepareGetIndex().addIndices(index).get().getSettings().get(index);
        assertThat(DataTier.TIER_PREFERENCE_SETTING.get(idxSettings), equalTo("data_content"));

        // index should be yellow
        ensureYellow(index);
    }

    public void testDataTierTelemetry() {
        startContentOnlyNode();
        startContentOnlyNode();
        startHotOnlyNode();

        client().admin()
            .indices()
            .prepareCreate(index)
            .setSettings(
                Settings.builder()
                    .put(DataTier.TIER_PREFERENCE, "data_hot")
                    .put("index.number_of_shards", 2)
                    .put("index.number_of_replicas", 0)
            )
            .setWaitForActiveShards(0)
            .get();

        client().admin()
            .indices()
            .prepareCreate(index + "2")
            .setSettings(Settings.builder().put("index.number_of_shards", 1).put("index.number_of_replicas", 1))
            .setWaitForActiveShards(0)
            .get();

        ensureGreen();
        client().prepareIndex(index, MapperService.SINGLE_MAPPING_NAME).setSource("foo", "bar").get();
        client().prepareIndex(index + "2", MapperService.SINGLE_MAPPING_NAME).setSource("foo", "bar").get();
        client().prepareIndex(index + "2", MapperService.SINGLE_MAPPING_NAME).setSource("foo", "bar").get();
        refresh(index, index + "2");

        DataTiersFeatureSetUsage usage = getUsage();
        // We can't guarantee that internal indices aren't created, so some of these are >= checks
        assertThat(usage.getTierStats().get(DataTier.DATA_CONTENT).nodeCount, equalTo(2));
        assertThat(usage.getTierStats().get(DataTier.DATA_CONTENT).indexCount, greaterThanOrEqualTo(1));
        assertThat(usage.getTierStats().get(DataTier.DATA_CONTENT).totalShardCount, greaterThanOrEqualTo(2));
        assertThat(usage.getTierStats().get(DataTier.DATA_CONTENT).primaryShardCount, greaterThanOrEqualTo(1));
        assertThat(usage.getTierStats().get(DataTier.DATA_CONTENT).docCount, greaterThanOrEqualTo(2L));
        assertThat(usage.getTierStats().get(DataTier.DATA_CONTENT).primaryByteCount, greaterThanOrEqualTo(1L));
        assertThat(usage.getTierStats().get(DataTier.DATA_CONTENT).primaryByteCountMedian, greaterThanOrEqualTo(1L));
        assertThat(usage.getTierStats().get(DataTier.DATA_CONTENT).primaryShardBytesMAD, greaterThanOrEqualTo(0L));
        assertThat(usage.getTierStats().get(DataTier.DATA_HOT).nodeCount, equalTo(1));
        assertThat(usage.getTierStats().get(DataTier.DATA_HOT).indexCount, greaterThanOrEqualTo(1));
        assertThat(usage.getTierStats().get(DataTier.DATA_HOT).totalShardCount, greaterThanOrEqualTo(2));
        assertThat(usage.getTierStats().get(DataTier.DATA_HOT).primaryShardCount, greaterThanOrEqualTo(2));
        assertThat(usage.getTierStats().get(DataTier.DATA_HOT).docCount, greaterThanOrEqualTo(1L));
        assertThat(usage.getTierStats().get(DataTier.DATA_HOT).primaryByteCount, greaterThanOrEqualTo(1L));
        assertThat(usage.getTierStats().get(DataTier.DATA_HOT).primaryByteCountMedian, greaterThanOrEqualTo(1L));
        assertThat(usage.getTierStats().get(DataTier.DATA_HOT).primaryShardBytesMAD, greaterThanOrEqualTo(0L));
    }

    public void testTierFilteringIgnoredByFilterAllocationDecider() {
        startContentOnlyNode();
        startHotOnlyNode();

        // Exclude all data_cold nodes
        client().admin()
            .cluster()
            .prepareUpdateSettings()
            .setTransientSettings(Settings.builder().put(DataTierAllocationDecider.CLUSTER_ROUTING_EXCLUDE, "data_cold").build())
            .get();

        // Create an index, which should be excluded just fine, ignored by the FilterAllocationDecider
        client().admin()
            .indices()
            .prepareCreate(index)
            .setSettings(Settings.builder().put("index.number_of_shards", 2).put("index.number_of_replicas", 0))
            .setWaitForActiveShards(0)
            .get();
    }

    public void testIllegalOnFrozen() {
        startDataNode();

        IllegalArgumentException e = expectThrows(
            IllegalArgumentException.class,
            () -> createIndex(
                index,
                Settings.builder()
                    .put("index.number_of_shards", 1)
                    .put("index.number_of_replicas", 0)
                    .put(DataTier.TIER_PREFERENCE, DataTier.DATA_FROZEN)
                    .build()
            )
        );
        assertThat(e.getMessage(), equalTo("[data_frozen] tier can only be used for partial searchable snapshots"));

        String initialTier = randomFrom(DataTier.DATA_HOT, DataTier.DATA_WARM, DataTier.DATA_COLD);
        createIndex(
            index,
            Settings.builder()
                .put("index.number_of_shards", 1)
                .put("index.number_of_replicas", 0)
                .put(DataTier.TIER_PREFERENCE, initialTier)
                .build()
        );

        IllegalArgumentException e2 = expectThrows(IllegalArgumentException.class, () -> updatePreference(DataTier.DATA_FROZEN));
        assertThat(e2.getMessage(), equalTo("[data_frozen] tier can only be used for partial searchable snapshots"));

        updatePreference(randomValueOtherThan(initialTier, () -> randomFrom(DataTier.DATA_HOT, DataTier.DATA_WARM, DataTier.DATA_COLD)));
    }

    private void updatePreference(String tier) {
        client().admin()
            .indices()
            .updateSettings(new UpdateSettingsRequest(index).settings(Map.of(DataTier.TIER_PREFERENCE, tier)))
            .actionGet();
    }

    private DataTiersFeatureSetUsage getUsage() {
        XPackUsageResponse usages = new XPackUsageRequestBuilder(client()).execute().actionGet();
        XPackFeatureSet.Usage dtUsage = usages.getUsages()
            .stream()
            .filter(u -> u instanceof DataTiersFeatureSetUsage)
            .collect(Collectors.toList())
            .get(0);
        if (dtUsage == null) {
            throw new IllegalArgumentException("no data tier usage found");
        }
        return (DataTiersFeatureSetUsage) dtUsage;
    }

    public void startDataNode() {
        Settings nodeSettings = Settings.builder().putList("node.roles", Arrays.asList("master", "data", "ingest")).build();
        internalCluster().startNode(nodeSettings);
    }

    public void startContentOnlyNode() {
        Settings nodeSettings = Settings.builder().putList("node.roles", Arrays.asList("master", "data_content", "ingest")).build();
        internalCluster().startNode(nodeSettings);
    }

    public void startHotOnlyNode() {
        Settings nodeSettings = Settings.builder().putList("node.roles", Arrays.asList("master", "data_hot", "ingest")).build();
        internalCluster().startNode(nodeSettings);
    }

    public void startWarmOnlyNode() {
        Settings nodeSettings = Settings.builder().putList("node.roles", Arrays.asList("master", "data_warm", "ingest")).build();
        internalCluster().startNode(nodeSettings);
    }

    public void startColdOnlyNode() {
        Settings nodeSettings = Settings.builder().putList("node.roles", Arrays.asList("master", "data_cold", "ingest")).build();
        internalCluster().startNode(nodeSettings);
    }

    public void startFrozenOnlyNode() {
        Settings nodeSettings = Settings.builder().putList("node.roles", Arrays.asList("master", "data_frozen", "ingest")).build();
        internalCluster().startNode(nodeSettings);
    }

    public void enforceDefaultTierPreference(boolean enforceDefaultTierPreference) {
        ClusterUpdateSettingsRequest request = new ClusterUpdateSettingsRequest();
        request.transientSettings(Settings.builder().put(DataTier.ENFORCE_DEFAULT_TIER_PREFERENCE, enforceDefaultTierPreference).build());
        assertAcked(client().admin().cluster().updateSettings(request).actionGet());
    }
}
