/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

package org.elasticsearch.client.transport;

import org.elasticsearch.action.admin.cluster.health.ClusterHealthRequest;
import org.elasticsearch.common.io.stream.NamedWriteable;
import org.elasticsearch.common.io.stream.NamedWriteableRegistry;
import org.elasticsearch.common.io.stream.NamedWriteableRegistry.Entry;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.env.Environment;
import org.elasticsearch.plugins.Plugin;
import org.elasticsearch.test.ESTestCase;
import org.elasticsearch.transport.MockTransportClient;
import org.elasticsearch.transport.TransportSettings;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutionException;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.hasItem;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.object.HasToString.hasToString;

public class TransportClientTests extends ESTestCase {

    public void testThatUsingAClosedClientThrowsAnException() throws ExecutionException, InterruptedException {
        final TransportClient client = new MockTransportClient(Settings.EMPTY);
        client.close();
        final IllegalStateException e = expectThrows(
            IllegalStateException.class,
            () -> client.admin().cluster().health(new ClusterHealthRequest()).actionGet()
        );
        assertThat(e, hasToString(containsString("transport client is closed")));
    }

    /**
     * test that when plugins are provided that want to register
     * {@link NamedWriteable}, those are also made known to the
     * {@link NamedWriteableRegistry} of the transport client
     */
    public void testPluginNamedWriteablesRegistered() {
        Settings baseSettings = Settings.builder().put(Environment.PATH_HOME_SETTING.getKey(), createTempDir()).build();
        try (TransportClient client = new MockTransportClient(baseSettings, Arrays.asList(MockPlugin.class))) {
            assertNotNull(client.namedWriteableRegistry.getReader(MockPlugin.MockNamedWriteable.class, MockPlugin.MockNamedWriteable.NAME));
        }
    }

    public void testSettingsContainsTransportClient() {
        final Settings baseSettings = Settings.builder().put(Environment.PATH_HOME_SETTING.getKey(), createTempDir()).build();
        try (TransportClient client = new MockTransportClient(baseSettings, Arrays.asList(MockPlugin.class))) {
            final Settings settings = TransportSettings.DEFAULT_FEATURES_SETTING.get(client.settings());
            assertThat(settings.keySet(), hasItem("transport_client"));
            assertThat(settings.get("transport_client"), equalTo("true"));
        }
    }

    public void testDefaultHeader() {
        final Settings baseSettings = Settings.builder().put(Environment.PATH_HOME_SETTING.getKey(), createTempDir()).build();
        try (TransportClient client = new MockTransportClient(baseSettings, Arrays.asList(MockPlugin.class))) {
            final ThreadContext threadContext = client.threadPool().getThreadContext();
            assertEquals("true", threadContext.getHeader("test"));
        }
    }

    public static class MockPlugin extends Plugin {

        @Override
        public List<Entry> getNamedWriteables() {
            return Arrays.asList(new Entry[] { new Entry(MockNamedWriteable.class, MockNamedWriteable.NAME, MockNamedWriteable::new) });
        }

        @Override
        public Settings additionalSettings() {
            return Settings.builder().put(ThreadContext.PREFIX + "." + "test", true).build();
        }

        public class MockNamedWriteable implements NamedWriteable {

            static final String NAME = "mockNamedWritable";

            MockNamedWriteable(StreamInput in) {}

            @Override
            public void writeTo(StreamOutput out) throws IOException {}

            @Override
            public String getWriteableName() {
                return NAME;
            }

        }
    }
}
