/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

package org.elasticsearch.ingest.useragent;

import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.common.Strings;
import org.elasticsearch.test.ESTestCase;
import org.junit.BeforeClass;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Field;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;

public class UserAgentProcessorFactoryTests extends ESTestCase {

    private static Map<String, UserAgentParser> userAgentParsers;

    private static String regexWithoutDevicesFilename = "regexes_without_devices.yml";
    private static Path userAgentConfigDir;

    @BeforeClass
    public static void createUserAgentParsers() throws IOException {
        Path configDir = createTempDir();
        userAgentConfigDir = configDir.resolve("ingest-user-agent");
        Files.createDirectories(userAgentConfigDir);

        // Copy file, leaving out the device parsers at the end
        try (
            BufferedReader reader = new BufferedReader(
                new InputStreamReader(UserAgentProcessor.class.getResourceAsStream("/regexes.yml"), StandardCharsets.UTF_8)
            );
            BufferedWriter writer = Files.newBufferedWriter(userAgentConfigDir.resolve(regexWithoutDevicesFilename));
        ) {
            String line;
            while ((line = reader.readLine()) != null) {
                if (line.startsWith("device_parsers:")) {
                    break;
                }

                writer.write(line);
                writer.newLine();
            }
        }

        userAgentParsers = IngestUserAgentPlugin.createUserAgentParsers(userAgentConfigDir, new UserAgentCache(1000));
    }

    public void testBuildDefaults() throws Exception {
        UserAgentProcessor.Factory factory = new UserAgentProcessor.Factory(userAgentParsers);

        Map<String, Object> config = new HashMap<>();
        config.put("field", "_field");

        String processorTag = randomAlphaOfLength(10);

        UserAgentProcessor processor = factory.create(null, processorTag, null, config);
        assertThat(processor.getTag(), equalTo(processorTag));
        assertThat(processor.getField(), equalTo("_field"));
        assertThat(processor.getUaParser().getUaPatterns().size(), greaterThan(0));
        assertThat(processor.getUaParser().getOsPatterns().size(), greaterThan(0));
        assertThat(processor.getUaParser().getDevicePatterns().size(), greaterThan(0));
        assertThat(processor.getProperties(), equalTo(EnumSet.allOf(UserAgentProcessor.Property.class)));
        assertFalse(processor.isExtractDeviceType());
        assertFalse(processor.isIgnoreMissing());
        assertTrue(processor.isUseECS());
    }

    public void testBuildWithIgnoreMissing() throws Exception {
        UserAgentProcessor.Factory factory = new UserAgentProcessor.Factory(userAgentParsers);

        Map<String, Object> config = new HashMap<>();
        config.put("field", "_field");
        config.put("ignore_missing", true);
        config.put("ecs", true);

        String processorTag = randomAlphaOfLength(10);

        UserAgentProcessor processor = factory.create(null, processorTag, null, config);
        assertThat(processor.getTag(), equalTo(processorTag));
        assertThat(processor.getField(), equalTo("_field"));
        assertThat(processor.getTargetField(), equalTo("user_agent"));
        assertThat(processor.getUaParser().getUaPatterns().size(), greaterThan(0));
        assertThat(processor.getUaParser().getOsPatterns().size(), greaterThan(0));
        assertThat(processor.getUaParser().getDevicePatterns().size(), greaterThan(0));
        assertThat(processor.getProperties(), equalTo(EnumSet.allOf(UserAgentProcessor.Property.class)));
        assertTrue(processor.isIgnoreMissing());
    }

    public void testBuildTargetField() throws Exception {
        UserAgentProcessor.Factory factory = new UserAgentProcessor.Factory(userAgentParsers);

        Map<String, Object> config = new HashMap<>();
        config.put("field", "_field");
        config.put("target_field", "_target_field");
        config.put("ecs", true);

        UserAgentProcessor processor = factory.create(null, null, null, config);
        assertThat(processor.getField(), equalTo("_field"));
        assertThat(processor.getTargetField(), equalTo("_target_field"));
    }

    public void testBuildRegexFile() throws Exception {
        UserAgentProcessor.Factory factory = new UserAgentProcessor.Factory(userAgentParsers);

        Map<String, Object> config = new HashMap<>();
        config.put("field", "_field");
        config.put("regex_file", regexWithoutDevicesFilename);
        config.put("ecs", true);

        UserAgentProcessor processor = factory.create(null, null, null, config);
        assertThat(processor.getField(), equalTo("_field"));
        assertThat(processor.getUaParser().getUaPatterns().size(), greaterThan(0));
        assertThat(processor.getUaParser().getOsPatterns().size(), greaterThan(0));
        assertThat(processor.getUaParser().getDevicePatterns().size(), equalTo(0));
    }

    public void testBuildExtractDeviceType() throws Exception {
        UserAgentProcessor.Factory factory = new UserAgentProcessor.Factory(userAgentParsers);
        boolean extractDeviceType = randomBoolean();

        Map<String, Object> config = new HashMap<>();
        config.put("field", "_field");
        config.put("extract_device_type", extractDeviceType);

        UserAgentProcessor processor = factory.create(null, null, null, config);
        assertThat(processor.getField(), equalTo("_field"));
        assertThat(processor.isExtractDeviceType(), equalTo(extractDeviceType));
    }

    public void testBuildNonExistingRegexFile() throws Exception {
        UserAgentProcessor.Factory factory = new UserAgentProcessor.Factory(userAgentParsers);

        Map<String, Object> config = new HashMap<>();
        config.put("field", "_field");
        config.put("regex_file", "does-not-exist.yml");

        ElasticsearchParseException e = expectThrows(ElasticsearchParseException.class, () -> factory.create(null, null, null, config));
        assertThat(e.getMessage(), equalTo("[regex_file] regex file [does-not-exist.yml] doesn't exist (has to exist at node startup)"));
    }

    public void testBuildFields() throws Exception {
        UserAgentProcessor.Factory factory = new UserAgentProcessor.Factory(userAgentParsers);

        Set<UserAgentProcessor.Property> properties = EnumSet.noneOf(UserAgentProcessor.Property.class);
        List<String> fieldNames = new ArrayList<>();
        int numFields = scaledRandomIntBetween(1, UserAgentProcessor.Property.values().length);
        Set<String> warnings = new HashSet<>();
        Set<UserAgentProcessor.Property> deprecated = Arrays.stream(UserAgentProcessor.Property.class.getFields())
            .filter(Field::isEnumConstant)
            .filter(field -> field.isAnnotationPresent(Deprecated.class))
            .map(field -> UserAgentProcessor.Property.valueOf(field.getName()))
            .collect(Collectors.toSet());
        for (int i = 0; i < numFields; i++) {
            UserAgentProcessor.Property property = UserAgentProcessor.Property.values()[i];
            if (deprecated.contains(property)) {
                warnings.add("the [" + property.name().toLowerCase(Locale.ROOT) + "] property is deprecated for the user-agent processor");
            }
            properties.add(property);
            fieldNames.add(property.name().toLowerCase(Locale.ROOT));
        }

        Map<String, Object> config = new HashMap<>();
        config.put("field", "_field");
        config.put("properties", fieldNames);
        config.put("ecs", true);

        UserAgentProcessor processor = factory.create(null, null, null, config);
        assertThat(processor.getField(), equalTo("_field"));
        assertThat(processor.getProperties(), equalTo(properties));
        if (warnings.size() > 0) {
            assertWarnings(warnings.toArray(Strings.EMPTY_ARRAY));
        }
    }

    public void testInvalidProperty() throws Exception {
        UserAgentProcessor.Factory factory = new UserAgentProcessor.Factory(userAgentParsers);

        Map<String, Object> config = new HashMap<>();
        config.put("field", "_field");
        config.put("properties", Collections.singletonList("invalid"));

        ElasticsearchParseException e = expectThrows(ElasticsearchParseException.class, () -> factory.create(null, null, null, config));
        assertThat(
            e.getMessage(),
            equalTo(
                "[properties] illegal property value [invalid]. valid values are [NAME, MAJOR, MINOR, "
                    + "PATCH, OS, OS_NAME, OS_MAJOR, OS_MINOR, DEVICE, BUILD, ORIGINAL, VERSION]"
            )
        );
    }

    public void testInvalidPropertiesType() throws Exception {
        UserAgentProcessor.Factory factory = new UserAgentProcessor.Factory(userAgentParsers);

        Map<String, Object> config = new HashMap<>();
        config.put("field", "_field");
        config.put("properties", "invalid");

        ElasticsearchParseException e = expectThrows(ElasticsearchParseException.class, () -> factory.create(null, null, null, config));
        assertThat(e.getMessage(), equalTo("[properties] property isn't a list, but of type [java.lang.String]"));
    }
}
