/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

package org.elasticsearch.client.security;

import org.elasticsearch.client.security.support.CertificateInfo;
import org.elasticsearch.test.ESTestCase;
import org.elasticsearch.test.EqualsHashCodeTestUtils;
import org.elasticsearch.xcontent.XContentBuilder;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static org.elasticsearch.test.AbstractXContentTestCase.xContentTester;

public class GetSslCertificatesResponseTests extends ESTestCase {
    public void testFromXContent() throws IOException {
        xContentTester(this::createParser, this::createTestInstance, this::toXContent, GetSslCertificatesResponse::fromXContent)
            .supportsUnknownFields(false)
            .test();
    }

    public void testEqualsAndHashCode() {
        final GetSslCertificatesResponse reponse = createTestInstance();
        EqualsHashCodeTestUtils.checkEqualsAndHashCode(reponse, this::copy, this::mutate);
    }

    protected GetSslCertificatesResponse createTestInstance() {
        final CertificateInfo info1 = new CertificateInfo(
            "certs/elastic-certificates.p12",
            "PKCS12",
            "instance",
            "CN=Elastic Certificate Tool Autogenerated CA",
            "a20f0ee901e8f69dc633ff633e5cd5437cdb4137",
            false,
            "2021-01-15T20:42:49.000Z"
        );
        final CertificateInfo info2 = new CertificateInfo(
            "certs/elastic-certificates.p12",
            "PKCS12",
            "ca",
            "CN=Elastic Certificate Tool Autogenerated CA",
            "a20f0ee901e8f69dc633ff633e5cd5437cdb4137",
            false,
            "2021-01-15T20:42:49.000Z"
        );
        final CertificateInfo info3 = new CertificateInfo(
            "certs/elastic-certificates.p12",
            "PKCS12",
            "instance",
            "CN=instance",
            "a20f0ee901e8f69dc633ff633e5cd5437cdb4137",
            true,
            "2021-01-15T20:44:32.000Z"
        );
        return new GetSslCertificatesResponse(Arrays.asList(info1, info2, info3));
    }

    private void toXContent(GetSslCertificatesResponse response, XContentBuilder builder) throws IOException {
        builder.startArray();
        for (CertificateInfo info : response.getCertificates()) {
            builder.startObject();
            builder.field(CertificateInfo.PATH.getPreferredName(), info.getPath());
            builder.field(CertificateInfo.FORMAT.getPreferredName(), info.getFormat());
            builder.field(CertificateInfo.ALIAS.getPreferredName(), info.getAlias());
            builder.field(CertificateInfo.SUBJECT_DN.getPreferredName(), info.getSubjectDn());
            builder.field(CertificateInfo.SERIAL_NUMBER.getPreferredName(), info.getSerialNumber());
            builder.field(CertificateInfo.HAS_PRIVATE_KEY.getPreferredName(), info.hasPrivateKey());
            builder.field(CertificateInfo.EXPIRY.getPreferredName(), info.getExpiry());
            builder.endObject();
        }
        builder.endArray();
    }

    private GetSslCertificatesResponse copy(GetSslCertificatesResponse original) {
        final List<CertificateInfo> infoList = new ArrayList<>(original.getCertificates());
        return new GetSslCertificatesResponse(infoList);
    }

    private GetSslCertificatesResponse mutate(GetSslCertificatesResponse original) {
        final int i = randomIntBetween(1, 5);
        final List<CertificateInfo> infoList = new ArrayList<>(original.getCertificates());
        switch (i) {
            case 1:
                infoList.remove(0);
                return new GetSslCertificatesResponse(infoList);
            case 2:
                final CertificateInfo info = new CertificateInfo(
                    "certs/elastic-certificates.crt",
                    "PEM",
                    "instance",
                    "CN=instance2",
                    "a20f0ee901e8f64t33ff633e5cd5437cdb4137",
                    true,
                    "2028-01-15T20:44:32.000Z"
                );
                infoList.add(info);
                return new GetSslCertificatesResponse(infoList);
            case 3:
                final CertificateInfo info2 = new CertificateInfo(
                    "certs/elastic-certificates.p12",
                    "PKCS12",
                    "instance",
                    "CN=instance1",
                    "a20f0ee901e8f69dc633ff633e5cd5437cdb4137",
                    true,
                    "2021-01-15T20:44:32.000Z"
                );
                infoList.remove(2);
                infoList.add(info2);
                return new GetSslCertificatesResponse(infoList);
            default:
                return new GetSslCertificatesResponse(Collections.emptyList());
        }
    }
}
